/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor;

import java.util.Map;

import woolpack.fn.Fn;

/**
 * 複合体を一段階分解した値を引数にして
 * {@link Visitor#visit(Object)}に委譲する関数です。
 * このクラスはエレメントの型によって
 * {@link MapAcceptor}、
 * {@link IterableAcceptor}、
 * {@link ArrayAcceptor}、
 * {@link BeanAcceptor}
 * を分岐する動作だけを行います。
 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
 * @author nakamura
 *
 * @param <E>
 */
public class ComplexAcceptor<E extends Exception> implements Fn<Visitor<?, E>, Void, E> {
	private final Fn<Visitor<?, E>, Void, E> mapAcceptor = new MapAcceptor<E>();
	private final Fn<Visitor<?, E>, Void, E> iterableAcceptor = new IterableAcceptor<E>();
	private final Fn<Visitor<?, E>, Void, E> arrayAcceptor = new ArrayAcceptor<E>();
	private final Fn<Visitor<?, E>, Void, E> beanAcceptor = new BeanAcceptor<E>();
	
	public Void exec(final Visitor<?, E> visitor) throws E {
		final Object object = visitor.getElement();
		if (object instanceof Map) {
			mapAcceptor.exec(visitor);
		} else if (object instanceof Iterable) {
			iterableAcceptor.exec(visitor);
		} else if (object.getClass().isArray()) {
			arrayAcceptor.exec(visitor);
		} else {
			beanAcceptor.exec(visitor);
		}
		return null;
	}
}
