/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.util.Map;

import woolpack.fn.Delegator;
import woolpack.fn.Fn;

/**
 * {@link Map}のビューに変換する関数です。
 * それ以外の場合はBeanとしてプロパティ名をキーにマップします。
 * 変換元の型がMapの場合は変換せず、
 * それ以外の場合はBeanとしてプロパティを{@link ConvertContext#getFn()}で変換してマップします。
 * 変換後の状態変化と変換前の状態変化は相互に伝播します。
 * <br/>適用しているデザインパターン：Chain of Responsibility。
 * @author nakamura
 * 
 */
public class ToMapViewConverter extends Delegator<ConvertContext, Void, RuntimeException> {

	/**
	 * @param fn {@link ConvertContext#getToType()}が{@link Map}でない場合の委譲先。
	 */
	public ToMapViewConverter(
			final Fn<? super ConvertContext, Void, ? extends RuntimeException> fn) {
		super(fn);
	}

	@Override
	public Void exec(final ConvertContext c) {
		if (!Map.class.isAssignableFrom(c.getToType())) {
			super.exec(c);
			return null;
		}
		if (c.getValue() instanceof Map) {
			return null;
		}
		c.setValue(new BeanMap(c.getValue(), c.getFn()));
		return null;
	}
}
