/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import javax.sql.DataSource;

import woolpack.fn.Fn;

public class SingleInput<C, R> implements Fn<C, R, Exception> {
	private DataSource dataSource;
	private Fn<? super C, ? extends String, ? extends Exception> queryFactory;
	private Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter;
	private Fn<? super SQLException, ?, ? extends Exception> errorFn;
	
	public SingleInput(
			final DataSource dataSource,
			final Fn<? super C, ? extends String, ? extends Exception> queryFactory,
			final Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		this.dataSource = dataSource;
		this.queryFactory = queryFactory;
		this.converter = converter;
		this.errorFn = errorFn;
	}

	public R exec(final C c) throws Exception {
		SQLException e0 = null;
		try {
			final Connection connection = dataSource.getConnection();
			try {
				final PreparedStatement statement = connection.prepareStatement(queryFactory.exec(c));
				try {
					statement.setObject(1, c);
					statement.execute();
					return converter.exec(statement);
				} catch (final SQLException e1) {
					errorFn.exec(e1);
					e0 = e1;
				} finally {
					statement.close();
				}
			} catch (final SQLException e1) {
				errorFn.exec(e1);
				if (e0 == null) {
					e0 = e1;
				}
			} finally {
				connection.close();
			}
		} catch (final SQLException e1) {
			errorFn.exec(e1);
			if (e0 == null) {
				e0 = e1;
			}
		}
		// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
		if (e0 != null) {
			throw e0;
		} else {
			return null;
		}
	}

	public DataSource getDataSource() {
		return dataSource;
	}
	public void setDataSource(final DataSource dataSource) {
		this.dataSource = dataSource;
	}
	public Fn<? super C, ? extends String, ? extends Exception> getQueryFactory() {
		return queryFactory;
	}
	public void setQueryFactory(final Fn<? super C, ? extends String, ? extends Exception> queryFactory) {
		this.queryFactory = queryFactory;
	}
	public Fn<? super PreparedStatement, ? extends R, ? extends Exception> getConverter() {
		return converter;
	}
	public void setConverter(final Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter) {
		this.converter = converter;
	}
	public Fn<? super SQLException, ?, ? extends Exception> getErrorFn() {
		return errorFn;
	}
	public void setErrorFn(final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		this.errorFn = errorFn;
	}
}
