/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import woolpack.test.TestBean;
import woolpack.test.TestUtils;

import junit.framework.TestCase;

public class TypeConvertUtilsTest extends TestCase {
	
	private static void scenario(final Object expected, final Object before, final Class toType, final String name) {
		final ConvertContext context = new ConvertContext();
		context.setToType(toType);
		context.setPropertyName(name);
		context.setValue(before);
		TypeConvertUtils.COLLECTION_CONVERTER.exec(context);
		assertTrue(TestUtils.equals(expected, context.getValue()));
	}

	private static void scenario(final Object expected, final Object before, final Class toType) {
		scenario(expected, before, toType, null);
	}
	
	private static java.util.Date toDate(final String s) throws ParseException {
		return new SimpleDateFormat("yyyyMMdd").parse(s);
	}

	public void testNormal() throws ParseException {
		scenario("c", 'c', String.class);
		scenario("true", Boolean.TRUE, String.class);
		scenario("false", Boolean.FALSE, String.class);
		scenario("20060708", toDate("20060708"), String.class);
		scenario("20060708", new java.sql.Date(toDate("20060708").getTime()), String.class);
		scenario("20060708", new java.sql.Time(toDate("20060708").getTime()), String.class);
		scenario("20060708", new java.sql.Timestamp(toDate("20060708").getTime()), String.class);
		scenario("1", (byte) 1, String.class);
		scenario("1,000", (short) 1000, String.class);
		scenario("1,000", (int) 1000, String.class);
		scenario("1,000", (long) 1000, String.class);
		scenario("1,000", (float) 1000, String.class);
		scenario("1,000", (double) 1000, String.class);
		scenario("1,000", BigInteger.valueOf(1000), String.class);
		scenario("1,000", new BigDecimal(1000), String.class);

		scenario(Character.valueOf('a'), "abc", Character.class);
		scenario(Boolean.TRUE, "true", Boolean.class);
		scenario(Boolean.FALSE, "false", Boolean.class);
		scenario(Boolean.FALSE, Boolean.FALSE, Boolean.class);
		
		scenario(toDate("20060708"), "20060708", java.util.Date.class);
		scenario(new java.sql.Date(toDate("20060708").getTime()), "20060708", java.sql.Date.class);
		scenario(new java.sql.Time(toDate("20060708").getTime()), "20060708", java.sql.Time.class);
		scenario(new java.sql.Timestamp(toDate("20060708").getTime()), "20060708", java.sql.Timestamp.class);

		scenario(Byte.valueOf((byte) 1), "1", Byte.class);
		scenario(Short.valueOf((short) 1), "1", Short.class);
		scenario(Integer.valueOf(1), "1", Integer.class);
		scenario(Long.valueOf(1), "1", Long.class);
		scenario(Float.valueOf(1), "1", Float.class);
		scenario(Double.valueOf(1), "1", Double.class);
		scenario(BigInteger.valueOf(1), "1", BigInteger.class);
		scenario(new BigDecimal(1), "1", BigDecimal.class);

		scenario(Byte.valueOf((byte) 1), Integer.valueOf(1), Byte.class);
		scenario(Short.valueOf((short) 1), Integer.valueOf(1), Short.class);
		scenario(Integer.valueOf(1), Long.valueOf(1), Integer.class);
		scenario(Long.valueOf(1), Integer.valueOf(1), Long.class);
		scenario(Float.valueOf(1), Integer.valueOf(1), Float.class);
		scenario(Double.valueOf(1), Integer.valueOf(1), Double.class);
		scenario(BigInteger.valueOf(1), Integer.valueOf(1), BigInteger.class);
		scenario(new BigDecimal(1), Integer.valueOf(1), BigDecimal.class);

		{
			final Object o = new Object();
			scenario(o, o, Object.class);
		}

		try {
			scenario(null, "abs", Integer.class);
			fail();
		} catch (final IllegalArgumentException e) {
			assertTrue(e.getCause() instanceof ParseException);
		}

		try {
			scenario(null, "", Character.class);
			fail();
		} catch (final IllegalArgumentException e) {
		}

		scenario(null, null, Object.class);
		scenario("1", "1", Object.class);
		scenario(new String[] { "1" }, new String[] { "1" }, Object.class);
		scenario(Arrays.asList("1"), Arrays.asList("1"), Object.class);

		scenario(null, null, Integer.class);
		scenario(Integer.valueOf(1), "1", Integer.class);
		scenario(Integer.valueOf(1), new String[] { "1" }, Integer.class);
		scenario(Integer.valueOf(1), new String[] { "1", "2" }, Integer.class);
		scenario(Integer.valueOf(1), Arrays.asList("1"), Integer.class);
		scenario(Integer.valueOf(1), Arrays.asList("1", "2"), Integer.class);

		scenario(null, null, int[].class);
		scenario(new int[] { 1 }, "1", int[].class);
		scenario(new int[] { 1 }, new String[] { "1" }, int[].class);
		scenario(new int[] { 1, 2 }, new String[] { "1", "2" }, int[].class);
		scenario(new int[] { 1 }, Arrays.asList("1"), int[].class);
		scenario(new int[] { 1, 2 }, Arrays.asList("1", "2"), int[].class);

		scenario(null, null, Integer[].class);
		scenario(new Integer[] { 1 }, "1", Integer[].class);
		scenario(new Integer[] { 1 }, new String[] { "1" }, Integer[].class);
		scenario(new Integer[] { 1, 2 }, new String[] { "1", "2" }, Integer[].class);
		scenario(new Integer[] { 1 }, Arrays.asList("1"), Integer[].class);
		scenario(new Integer[] { 1, 2 }, Arrays.asList("1", "2"), Integer[].class);

		scenario(null, null, Collection.class);
		scenario(Arrays.asList("1"), "1", Collection.class);
		scenario(Arrays.asList("1"), new String[] { "1" }, Collection.class);
		scenario(Arrays.asList("1", "2"), new String[] { "1", "2" }, Collection.class);
		scenario(Arrays.asList("1"), Arrays.asList("1"), Collection.class);
		scenario(Arrays.asList("1", "2"), Arrays.asList("1", "2"), Collection.class);

		scenario(null, null, List.class);
		scenario(Arrays.asList("1"), "1", List.class);
		scenario(Arrays.asList("1"), new String[] { "1" }, List.class);
		scenario(Arrays.asList("1", "2"), new String[] { "1", "2" }, List.class);
		scenario(Arrays.asList("1"), Arrays.asList("1"), List.class);
		scenario(Arrays.asList("1", "2"), Arrays.asList("1", "2"), List.class);

		{
			final ConvertContext context = new ConvertContext();
			context.setPropertyName(null);
			context.setToType(int[].class);
			context.setValue(new String[]{"1", "a"});
			try {
				TypeConvertUtils.COLLECTION_CONVERTER.exec(context);
				fail();
			} catch(final RuntimeException e) {
			}
			assertTrue(TestUtils.equals(new String[]{"1", "a"}, context.getValue()));
		}
		
		{
			final TestBean bean = new TestBean();
			final ConvertContext context = new ConvertContext();
			context.setPropertyName(null);
			context.setToType(Map.class);
			context.setValue(bean);
			TypeConvertUtils.COLLECTION_CONVERTER.exec(context);
			final Map<String, Object> map = (Map<String, Object>) context.getValue();
			
			bean.setMyInt(3);
			assertEquals(Integer.valueOf(3), map.get("myInt"));

			map.put("myInt", 5);
			assertEquals(5, bean.getMyInt());

			map.put("myIntArray", Arrays.asList("2", "4"));
			assertTrue(TestUtils.equals(new int[]{2, 4}, bean.getMyIntArray()));
		}
		{
			final Map<String, Object> map = new HashMap<String, Object>();
			final ConvertContext context = new ConvertContext();
			context.setPropertyName(null);
			context.setToType(Map.class);
			context.setValue(map);
			TypeConvertUtils.COLLECTION_CONVERTER.exec(context);
			assertSame(map, context.getValue());
		}
		{
			final Map<String, Object> map = new HashMap<String, Object>();
			map.put("myInt", "3");
			map.put("myIntArray", Arrays.asList("2", "4"));
			
			final ConvertContext context = new ConvertContext();
			context.setPropertyName(null);
			context.setToType(TestBean.class);
			context.setValue(map);
			TypeConvertUtils.COLLECTION_CONVERTER.exec(context);
			final TestBean bean = (TestBean) context.getValue();
			
			assertEquals(3, bean.getMyInt());
			assertTrue(TestUtils.equals(new int[]{2, 4}, bean.getMyIntArray()));
		}
	}
	
	public void testGetPropertyName() {
		final ConvertContext context = new ConvertContext();
		context.setPropertyName("hoge");
		assertEquals("hoge", TypeConvertUtils.GET_PROPERTY_NAME.exec(context));
	}
}
