/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.test;

import java.util.Arrays;
import java.util.HashSet;

import javax.xml.transform.TransformerException;

import junit.framework.TestCase;

import org.w3c.dom.Node;

import woolpack.utils.Utils;
import woolpack.xml.XmlTransformerUtils;

public class TestUtilsTest extends TestCase {

	public void testQuery() {
		assertEquals(Utils
						.map("name0", Arrays.asList("value00", "value01"))
						.map("name1", Arrays.asList("value10", "value11"))
						.map("name2", Arrays.asList("value20", "value21")),
						TestUtils.selectQuery("hoge.html?name0=value00&name1=value10&name2=value20&name0=value01&name1=value11&name2=value21"));
	}

	public void testSelectForm() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value00</TEXTAREA>"
				+ "<INPUT name=\"name1\" value=\"value10\" />"
				+ "<SPAN id=\"name2\">value20</SPAN>"
				+ "<TEXTAREA name=\"name0\" >value01</TEXTAREA>"
				+ "<INPUT name=\"name1\" value=\"value11\" />"
				+ "<SPAN id=\"name2\">value21</SPAN>"
				+ "<INPUT type=\"radio\" name=\"name3\" value=\"value30\" />"
				+ "<INPUT type=\"radio\" name=\"name3\" value=\"value31\" checked=\"true\" />"
				+ "<INPUT type=\"checkbox\" name=\"name4\" value=\"value40\" />"
				+ "<INPUT type=\"checkbox\" name=\"name4\" value=\"value41\" checked=\"true\" />"
				+ "<SELECT name=\"name5\">"
				+ "<OPTION value=\"value50\" />label50"
				+ "<OPTION value=\"value51\" selected=\"true\" />label51"
				+ "</SELECT>"
				+ "<BUTTON name=\"name6\" value=\"value61\" />"
				+ "</BODY></HTML>");
		assertEquals(
				Utils
				.map("name0", Arrays.asList("value00", "value01"))
				.map("name1", Arrays.asList("value10", "value11"))
				.map("name3", Arrays.asList("value31"))
				.map("name4", Arrays.asList("value41"))
				.map("name5", Arrays.asList("value51"))
				.map("name6", Arrays.asList("value61")),
				TestUtils.selectForm(node));
	}

	public void testSelectEmbedded() throws TransformerException {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value00</TEXTAREA>"
				+ "<INPUT name=\"name1\" value=\"value10\" />"
				+ "<SPAN id=\"name2\">value20</SPAN>"
				+ "<TEXTAREA name=\"name0\" >value01</TEXTAREA>"
				+ "<INPUT name=\"name1\" value=\"value11\" />"
				+ "<SPAN id=\"name2\">value21</SPAN>"
				+ "<INPUT type=\"radio\" name=\"name3\" value=\"value30\" />"
				+ "<INPUT type=\"radio\" name=\"name3\" value=\"value31\" checked=\"true\" />"
				+ "<INPUT type=\"checkbox\" name=\"name4\" value=\"value40\" />"
				+ "<INPUT type=\"checkbox\" name=\"name4\" value=\"value41\" checked=\"true\" />"
				+ "<SELECT name=\"name5\">"
				+ "<OPTION value=\"value50\" />label50"
				+ "<OPTION value=\"value51\" selected=\"true\" />label51"
				+ "</SELECT>"
				+ "</BODY></HTML>");
		assertEquals(
				Utils.map("name2", Arrays.asList("value20", "value21")),
				TestUtils.selectEmbedded(Arrays.asList("id"), node));
	}

	private void checkEquals(final boolean expected, final Object o0, final Object o1) {
		assertEquals(expected, TestUtils.equals(o0, o1));
	}

	public void testEquals() {
		checkEquals(true, null, null);
		checkEquals(false, 1, null);
		checkEquals(false, null, 1);

		checkEquals(true, 1, 1);
		checkEquals(false, 1, 2);

		checkEquals(true, Arrays.asList(1, 2), Arrays.asList(1, 2));
		checkEquals(false, Arrays.asList(1, 2), Arrays.asList(2, 1));
		checkEquals(false, Arrays.asList(1, 2), Arrays.asList(1, 3));
		checkEquals(false, Arrays.asList(1, 2), Arrays.asList(1));

		checkEquals(true, new HashSet<Integer>(Arrays.asList(1, 2)), Arrays.asList(1, 2));
		checkEquals(true, new HashSet<Integer>(Arrays.asList(1, 2)), Arrays.asList(2, 1));
		checkEquals(false, new HashSet<Integer>(Arrays.asList(1, 2)), Arrays.asList(1, 3));
		checkEquals(false, new HashSet<Integer>(Arrays.asList(1, 2)), Arrays.asList(1));

		checkEquals(true, 
				Utils.linkedMap(1, 11).map(2, 12),
				Utils.linkedMap(1, 11).map(2, 12));
		checkEquals(false,
				Utils.linkedMap(1, 11).map(2, 12),
				Utils.linkedMap(2, 12).map(1, 11));
		checkEquals(false,
				Utils.linkedMap(1, 11).map(2, 12),
				Utils.linkedMap(1, 11).map(2, 13));
		checkEquals(false,
				Utils.linkedMap(1, 11).map(2, 12),
				Utils.linkedMap(1, 11).map(3, 12));
		checkEquals(false,
				Utils.linkedMap(1, 11).map(2, 12),
				Utils.linkedMap(1, 11));
		checkEquals(false,
				Utils.linkedMap(1, 11),
				Utils.linkedMap(1, 11).map(2, 12));

		checkEquals(true,
				Utils.map(1, 11).map(2, 12),
				Utils.map(1, 11).map(2, 12));
		checkEquals(true,
				Utils.map(1, 11).map(2, 12),
				Utils.map(2, 12).map(1, 11));
		checkEquals(false,
				Utils.map(1, 11).map(2, 12),
				Utils.map(1, 11).map(2, 13));
		checkEquals(false,
				Utils.map(1, 11).map(2, 12),
				Utils.map(1, 11).map(3, 12));
		checkEquals(false,
				Utils.map(1, 11).map(2, 12),
				Utils.map(1, 11));
		checkEquals(false, 
				Utils.map(1, 11),
				Utils.map(1, 11).map(2, 12));

		checkEquals(false, Utils.map(1, 11), 1);
		checkEquals(false, 1, Utils.map(1, 11));

		checkEquals(false, Arrays.asList(1), 1);
		checkEquals(false, 1, Arrays.asList(1));

		checkEquals(false, Utils.map(1, 11), Arrays.asList(1));
		checkEquals(false, Arrays.asList(1), Utils.map(1, 11));

		checkEquals(true,
				Utils.map(1, Arrays.asList(1, 2)),
				Utils.map(1, Arrays.asList(1, 2)));
		checkEquals(false, Utils.map(1, Arrays.asList(1, 2)),
				Utils.map(1, Arrays.asList(2, 1)));
		checkEquals(false,
				Utils.map(1, Arrays.asList(1, 2)),
				Utils.map(1, Arrays.asList(1, 3)));
		checkEquals(false,
				Utils.map(1, Arrays.asList(1, 2)),
				Utils.map(1, Arrays.asList(1)));

		checkEquals(true,
				Utils.map(1, new HashSet<Integer>(Arrays.asList(1, 2))),
				Utils.map(1, Arrays.asList(1, 2)));
		checkEquals(true,
				Utils.map(1, new HashSet<Integer>(Arrays.asList(1, 2))),
				Utils.map(1, Arrays.asList(2, 1)));
		checkEquals(false, Utils.map(1, new HashSet<Integer>(Arrays.asList(1, 2))),
				Utils.map(1, Arrays.asList(1, 3)));
		checkEquals(false,
				Utils.map(1, new HashSet<Integer>(Arrays.asList(1, 2))),
				Utils.map(1, Arrays.asList(1)));
	}
}
