/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.xml.transform.TransformerException;

import junit.framework.TestCase;
import woolpack.el.PathEL;
import woolpack.utils.Utils;
import woolpack.xml.MapNodeContext;
import woolpack.xml.XmlTransformerUtils;
import woolpack.xml.XmlUtils;

public class AutoUpdaterTest extends TestCase {
	public void testSimple() throws TransformerException {
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"
				+ "<DIV id=\"div0\" >value0</DIV>"
				+ "<SPAN id=\"span0\" >value0</SPAN>"
				+ "<INPUT name=\"input0\" value=\"value0\" />"
				+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"
				+ "</BODY></HTML>"));
		context.setMap(Utils
				.map("textarea0", "textareavalue0")
				.map("div0", "divvalue0")
				.map("span0", "spanvalue0")
				.map("input0", "inputvalue0"));
		
		HtmlUtils.updateAuto(Arrays.asList("name", "id"), new PathEL("map"), null)
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"textarea0\" >textareavalue0</TEXTAREA>"
				+ "<DIV id=\"div0\" >divvalue0</DIV>"
				+ "<SPAN id=\"span0\" >spanvalue0</SPAN>"
				+ "<INPUT name=\"input0\" value=\"inputvalue0\" />"
				+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testBeanIterate() throws TransformerException {
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>"));
		context.setMap(Utils
				.map("iterate0", Utils
						.list(Utils.map("id0", "value0"))
						.list(Utils.map("id0", "value1"))));
		
		HtmlUtils.updateAuto(Arrays.asList("name", "id"), new PathEL("map"), null)
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value0</TD></TR>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value1</TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testAtomIterate() throws TransformerException {
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"span0\" >spanvalue0</SPAN>"
				+ "</BODY></HTML>"));
		context.setMap(Utils
				.map("span0", Utils
						.list("spanvalue0")
						.list("spanvalue1")));
		
		HtmlUtils.updateAuto(Arrays.asList("name", "id"), new PathEL("map"), null)
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"span0\" >spanvalue0</SPAN>"
				+ "<SPAN id=\"span0\" >spanvalue1</SPAN>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testIterateEmpty() throws TransformerException {
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>"));
		context.setMap(Utils.map("iterate0", new HashMap[] {}));
		
		HtmlUtils.updateAuto(Arrays.asList("name", "id"), new PathEL("map"), null)
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE></TABLE>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testArray() throws TransformerException {
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>"));
		context.setMap(Utils.map("iterate0", new Map[] {
				Utils.map("id0", "value0"),
				Utils.map("id0", "value1"), }));
		
		HtmlUtils.updateAuto(Arrays.asList("name", "id"), new PathEL("map"), null)
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value0</TD></TR>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value1</TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testNotFound() throws TransformerException {
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"
				+ "<DIV id=\"div0\" >value0</DIV>"
				+ "<SPAN id=\"span0\" >value0</SPAN>"
				+ "<INPUT name=\"input0\" value=\"value0\" />"
				+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"
				+ "</BODY></HTML>"));
		context.setMap(Utils.map("iterate0", new Object()));
		
		HtmlUtils.updateAuto(Arrays.asList("name", "id"), new PathEL("map.iterate0"), null)
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"
				+ "<DIV id=\"div0\" >value0</DIV>"
				+ "<SPAN id=\"span0\" >value0</SPAN>"
				+ "<INPUT name=\"input0\" value=\"value0\" />"
				+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testComplex() throws TransformerException {
		final MapNodeContext context = new MapNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"span00\" >value0</SPAN>"
				+ "<SPAN id=\"myMap\" >"
				+ "<SPAN id=\"span10\" >value0</SPAN>"
				+ "<SPAN id=\"span11\" >value0</SPAN>"
				+ "</SPAN>"
				+ "<SPAN id=\"span01\" >value0</SPAN>"
				+ "</BODY></HTML>"));
		context.setMap(Utils
				.map("myMap", (Object) Utils
						.map("span10", "spanvalue10")
						.map("span11", "spanvalue11"))
				.map("span00", "spanvalue00")
				.map("span01", "spanvalue01"));
		
		HtmlUtils.updateAuto(Arrays.asList("name", "id"), new PathEL("map"), null)
		.exec(context);
		
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"span00\" >spanvalue00</SPAN>"
				+ "<SPAN id=\"myMap\" >"
				+ "<SPAN id=\"span10\" >spanvalue10</SPAN>"
				+ "<SPAN id=\"span11\" >spanvalue11</SPAN>"
				+ "</SPAN>"
				+ "<SPAN id=\"span01\" >spanvalue01</SPAN>"
				+ "</BODY></HTML>"), context.getNode()));
	}
}
