/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.fn.Fn;

/**
 * ノードを検索して委譲する{@link Fn}です。
 * 委譲先でノードを操作した場合、次の検索結果に影響するかは委譲先に依存します。
 * 
 * @author nakamura
 * 
 * @param <C>
 * @param <E>
 */
public class NodeFinder<C extends NodeContext, E extends Exception> implements Fn<C, Void, E> {
	private Fn<? super Node, ? extends NodeList, ? extends E> findable;
	private Fn<? super C, Void, ? extends E> firstFn;
	private Fn<? super C, Void, ? extends E> pluralFn;

	/**
	 * @param findable
	 * @param firstFn 最初の委譲先(ポインタは検索結果)。
	 * @param pluralFn 2番目以降の委譲先(ポインタは検索結果)。
	 */
	public NodeFinder(
			final Fn<? super Node, ? extends NodeList, ? extends E> findable,
			final Fn<? super C, Void, ? extends E> firstFn,
			final Fn<? super C, Void, ? extends E> pluralFn) {
		this.findable = findable;
		this.firstFn = firstFn;
		this.pluralFn = pluralFn;
	}

	public Void exec(final C context) throws E {
		final Node base = context.getNode();
		final NodeList nodeList = findable.exec(base);
		final Node[] beforeNodeList = new Node[nodeList.getLength()];
		for (int i = 0; i < nodeList.getLength(); i++) {
			beforeNodeList[i] = nodeList.item(i);
		}
		try {
			for (int i = 0; i < beforeNodeList.length; i++) {
				context.setNode(beforeNodeList[i]);
				if (i == 0) {
					firstFn.exec(context);
				} else {
					pluralFn.exec(context);
				}
			}
		} finally {
			context.setNode(base);
		}
		return null;
	}

	public Fn<? super Node, ? extends NodeList, ? extends E> getFindable() {
		return findable;
	}
	public void setFindable(final Fn<? super Node, ? extends NodeList, ? extends E> findable) {
		this.findable = findable;
	}
	public Fn<? super C, Void, ? extends E> getFirstFn() {
		return firstFn;
	}
	public void setFirstFn(final Fn<? super C, Void, ? extends E> firstFn) {
		this.firstFn = firstFn;
	}
	public Fn<? super C, Void, ? extends E> getPluralFn() {
		return pluralFn;
	}
	public void setPluralFn(final Fn<? super C, Void, ? extends E> pluralFn) {
		this.pluralFn = pluralFn;
	}
}
