/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import org.w3c.dom.Element;

import woolpack.fn.Fn;

/**
 * コンテキストを DOM エレメントとして
 * 属性名に対応する値をキーとして委譲先を検索し委譲する{@link Fn}です。
 * 
 * @author nakamura
 * 
 * @param <C>
 * @param <E>
 */
public class AttrValueBranch<C extends NodeContext, E extends Exception> implements Fn<C, Void, E> {
	private Iterable<String> attrNames;
	private Fn<String, ? extends Fn<? super C, Void, ? extends E>, ? extends E> fn;

	/**
	 * @param attrNames 属性値を取得するための属性名の候補一覧。
	 * @param fn 属性値と委譲先の対応表。
	 */
	public AttrValueBranch(
			final Iterable<String> attrNames,
			final Fn<String, ? extends Fn<? super C, Void, ? extends E>, ? extends E> fn) {
		this.attrNames = attrNames;
		this.fn = fn;
	}

	public Void exec(final C context) throws E {
		final Element element = (Element) context.getNode();
		for (final String attrName : attrNames) {
			final String value = element.getAttribute(attrName);
			final Fn<? super C, Void, ? extends E> fn2 = fn.exec(value);
			if (fn2 != null) {
				return fn2.exec(context);
			}
		}
		return null;
	}

	public Iterable<String> getAttrNames() {
		return attrNames;
	}
	public void setAttrNames(final Iterable<String> attrNames) {
		this.attrNames = attrNames;
	}
	public Fn<String, ? extends Fn<? super C, Void, ? extends E>, ? extends E> getFn() {
		return fn;
	}
	public void setFn(final Fn<String, ? extends Fn<? super C, Void, ? extends E>, ? extends E> fn) {
		this.fn = fn;
	}
}
