/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import woolpack.bool.BoolUtils;
import woolpack.bool.BooleanOperator;
import woolpack.fn.Fn;
import woolpack.utils.Utils;

/**
 * ユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 * 
 */
public final class ValidatorUtils {
	public static final ValueGetter<RuntimeException> VALUE = new ValueGetter<RuntimeException>();
	public static final MapGetter<RuntimeException> MAP = new MapGetter<RuntimeException>();
	public static final ValuesGetter<RuntimeException> VALUES = new ValuesGetter<RuntimeException>();

	private ValidatorUtils() {
	}

	/**
	 * 引数のキーが全て文字列型・値が全てオブジェクトの一覧とみなして変換します。
	 * サーブレットAPIの request.getParameterMap() を変換するために定義しています。
	 * 返却値を更新しても引数には影響しません。
	 * @param map
	 * @return 変換結果。
	 */
	public static Map<String, List<Object>> convert(final Map map) {
		final Map<String, List<Object>> map1 = new HashMap<String, List<Object>>();
		for (final Object entryObject : map.entrySet()) {
			final Entry entry = (Entry) entryObject;
			final Iterable c = Utils.toIterable(entry.getValue());
			final List<Object> list = new ArrayList<Object>();
			for (final Object o : c) {
				list.add(o);
			}
			map1.put((String) entry.getKey(), list);
		}
		return map1;
	}
	
	public static <E extends Exception> NameBranchIfExists<E> branchByNameIfExists(
			final BooleanOperator<ValidatorContext, E> operator,
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean, ? extends E>> map) {
		return new NameBranchIfExists<E>(operator, map);
	}
	
	/**
	 * {@link BoolUtils#and()}を使用します。
	 * @param <E>
	 * @param map プロパティ名と委譲先の対応表。
	 * @return 関数。
	 */
	public static <E extends Exception> NameBranchIfExists<E> branchByNameIfExists(
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean, ? extends E>> map) {
		return new NameBranchIfExists<E>(BoolUtils.<ValidatorContext, E>and(), map);
	}
	
	public static <E extends Exception> NameBranch<E> branchByName(
			final BooleanOperator<ValidatorContext, E> operator,
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean, ? extends E>> map) {
		return new NameBranch<E>(operator, map);
	}
	
	/**
	 * {@link BoolUtils#and()}を使用します。
	 * @param <E>
	 * @param map プロパティ名と委譲先の対応表。
	 * @return 関数。
	 */
	public static <E extends Exception> NameBranch<E> branchByName(
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean, ? extends E>> map) {
		return new NameBranch<E>(BoolUtils.<ValidatorContext, E>and(), map);
	}
	
	public static <E extends Exception> ConvertValidator<E> convertValue(final Fn<Object, ?, ? extends E> fn) {
		return new ConvertValidator<E>(fn);
	}
	
	public static MessageValidator<RuntimeException> message(final String message) {
		return new MessageValidator<RuntimeException>(message);
	}
	
	public static <E extends Exception> LocalIndexValidator<E> localIndex(
			final int tmpIndex,
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		return new LocalIndexValidator<E>(tmpIndex, fn);
	}
	
	public static <E extends Exception> LocalKeyValidator<E> localKey(
			final String tmpKey,
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		return new LocalKeyValidator<E>(tmpKey, fn);
	}
	
	public static <E extends Exception> ValueLoopValidator<E> loopValue(
			final BooleanOperator<ValidatorContext, E> operator,
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		return new ValueLoopValidator<E>(operator, fn);
	}
	
	public static <E extends Exception> ValueLoopValidator<E> loopValue(
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		return new ValueLoopValidator<E>(fn);
	}
}
