/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import woolpack.fn.Fn;
import woolpack.utils.AbstractKeyIteratorMap;

/**
 * Beanを{@link Map}に変換するアダプタです。
 * <br/>適用しているデザインパターン：Adapter。
 * @author nakamura
 *
 */
public class BeanMap extends AbstractKeyIteratorMap<String, Object> {
	private final Object object;
	private final Fn<ConvertContext, Void, ? extends RuntimeException> fn;
	private final Map<String, PropertyDescriptor> map;
	private final Set<String> set;
	
	/**
	 * 型変換に{@link TypeConvertUtils#COLLECTION_CONVERTER}を使用します。
	 * @param object bean。
	 */
	public BeanMap(final Object object) {
		this(object, TypeConvertUtils.COLLECTION_CONVERTER);
	}
	
	/**
	 * 
	 * @param object bean。
	 * @param fn 型変換用の{@link Fn}。
	 */
	public BeanMap(final Object object, final Fn<ConvertContext, Void, ? extends RuntimeException> fn) {
		this.object = object;
		this.fn = fn;
		final BeanInfo beanInfo;
		try {
			beanInfo = Introspector.getBeanInfo(object.getClass());
		} catch (final IntrospectionException e) {
			throw new IllegalArgumentException(e);
		}
		final PropertyDescriptor[] array = beanInfo.getPropertyDescriptors();
		map = new HashMap<String, PropertyDescriptor>(array.length);
		for (final PropertyDescriptor p : array) {
			final String key = p.getName();
			if ("class".equals(key)) {
				continue;
			}
			map.put(key, p);
		}
		set = Collections.unmodifiableSet(map.keySet());
	}

	@Override
	protected Iterator<String> getKeyIterator() {
		return set.iterator();
	}

	@Override
	protected Object getValue(final Object key) {
		final PropertyDescriptor p = map.get(key);
		if (p != null) {
			final Method m = p.getReadMethod();
			if (m != null) {
				try {
					return m.invoke(object, new Object[0]);
				} catch (final IllegalAccessException e) {
					throw new IllegalStateException(e);
				} catch (final InvocationTargetException e) {
					throw new IllegalStateException(e);
				}
			}
		}
		throw new IllegalStateException("property not found.");
	}

	@Override
	public Object put(final String key, final Object value) {
		final PropertyDescriptor p = map.get(key);
		if (p != null) {
			final Method m = p.getWriteMethod();
			if (m != null) {
				try {
					final ConvertContext c = new ConvertContext();
					c.setPropertyName(key);
					c.setToType(p.getPropertyType());
					c.setValue(value);
					fn.exec(c);
					return m.invoke(object, new Object[]{c.getValue()});
				} catch (final IllegalAccessException e) {
					throw new IllegalStateException(e);
				} catch (final InvocationTargetException e) {
					throw new IllegalStateException(e);
				}
			}
		}
		throw new IllegalStateException("property not found.");
	}
}
