/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.tx;

import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import woolpack.fn.Fn;

/**
 * RDBとのトランザクションのブロックを処理する{@link Fn}です。
 * {@link TxBuilder#getTxDataSource()}と組み合わせて使用します。
 * {@link #exec(Object)}は{@link DataSource#getConnection()}を実行して
 * 処理を委譲先に委譲します。
 * 委譲先から{@link RuntimeException}を受け取った場合は{@link Connection#rollback()}を呼び出します。
 * 最後に{@link Connection#close()}を呼び出します。
 * このクラスのメソッドは最初に発生した例外を再スローするため、メソッド内で続けて発生した例外の情報は失われます。
 * <br/>適用しているデザインパターン：Before After。
 * @author nakamura
 *
 * @param <C>
 * @param <R>
 */
public class TxFn<C, R> implements Fn<C, R, Exception> {
	private DataSource dataSource;
	private Fn<? super C, ? extends R, ? extends Exception> fn;
	
	public TxFn(final DataSource dataSource, final Fn<? super C, ? extends R, ? extends Exception> fn) {
		this.dataSource = dataSource;
		this.fn = fn;
	}

	public R exec(final C c) throws Exception {
		Exception e0 = null;
		try {
			final Connection connection = dataSource.getConnection();
			try {
				return fn.exec(c);
			} catch (final Exception e1) {
				e0 = e1;
				connection.rollback();
			} finally {
				connection.close();
			}
		} catch (final SQLException e1) {
			if (e0 == null) {
				e0 = e1;
			}
		}
		if (e0 != null) {
			throw e0;
		} else {
			return null;
		}
	}

	public DataSource getDataSource() {
		return dataSource;
	}
	public void setDataSource(final DataSource dataSource) {
		this.dataSource = dataSource;
	}
	public Fn<? super C, ? extends R, ? extends Exception> getFn() {
		return fn;
	}
	public void setFn(final Fn<? super C, ? extends R, ? extends Exception> fn) {
		this.fn = fn;
	}
}
