/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.fn;

/**
 * Java の try-catch-finally のそれぞれのブロックで委譲する{@link Fn}です。
 * @author nakamura
 * 
 * @param <C>
 * @param <R>
 * @param <E>
 */
public class TryFn<C, R, E extends Exception> implements Fn<C, R, E> {
	private Fn<? super C, ? extends R, ? extends Exception> fn;
	private Fn<? super Throwable, ? extends R, ? extends E> reportFn;
	private Fn<? super C, ?, ? extends E> finallyFn;

	/**
	 * @param fn try ブロックにおける委譲先。
	 * @param reportFn catch ブロックにおける委譲先。
	 * @param finallyFn finally ブロックにおける委譲先。
	 */
	public TryFn(
			final Fn<? super C, ? extends R, ? extends Exception> fn,
			final Fn<? super Throwable, ? extends R, ? extends E> reportFn,
			final Fn<? super C, ?, ? extends E> finallyFn) {
		this.fn = fn;
		this.reportFn = reportFn;
		this.finallyFn = finallyFn;
	}

	public R exec(final C c) throws E {
		try {
			return fn.exec(c);
		} catch (final Exception e) {
			return reportFn.exec(e);
		} finally {
			finallyFn.exec(c);
		}
	}

	public Fn<? super Throwable, ? extends R, ? extends E> getReportFn() {
		return reportFn;
	}
	public void setReportFn(final Fn<? super Throwable, ? extends R, ? extends E> reportFn) {
		this.reportFn = reportFn;
	}
	public Fn<? super C, ?, ? extends E> getFinallyFn() {
		return finallyFn;
	}
	public void setFinallyFn(final Fn<? super C, ?, ? extends E> finallyFn) {
		this.finallyFn = finallyFn;
	}
	public Fn<? super C, ? extends R, ? extends Exception> getFn() {
		return fn;
	}
	public void setFn(final Fn<? super C, ? extends R, ? extends Exception> fn) {
		this.fn = fn;
	}
}
