/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.fn;

import java.util.Collection;
import java.util.Map;

import woolpack.utils.Utils;

/**
 * ユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドを含みます。
 * 
 * @author nakamura
 *
 */
public final class FnUtils {
	
	/**
	 * プリミティブ型をオブジェクト型に変換する{@link Fn}です。
	 */
	public static final Fn<Class, Class, RuntimeException> TO_WRAPPER = switching(
			Utils.<Class, Class>
			map(boolean.class, Boolean.class)
			.map(char.class, Character.class)
			.map(byte.class, Byte.class)
			.map(short.class, Short.class)
			.map(int.class, Integer.class)
			.map(long.class, Long.class)
			.map(float.class, Float.class)
			.map(double.class, Double.class), FnUtils.<Class>echo());
	
	private FnUtils() {
	}
	
	public static <R> CastFn<R, RuntimeException> castTo(final Class<R> clazz) {
		return new CastFn<R, RuntimeException>(clazz);
	}
	
	public static <C, R, E extends Exception> ExecFn<C, R, E> exec(
			final Fn<? super C, ? extends Fn<? super C, ? extends R, ? extends E>, ? extends E> fn) {
		return new ExecFn<C, R, E>(fn);
	}
	
	public static <C> EchoFn<C, RuntimeException> echo() {
		return new EchoFn<C, RuntimeException>();
	}
	
	public static <C, R, E extends Exception> FixFn<C, R, E> fixThrows(final R value) {
		return new FixFn<C, R, E>(value);
	}
	
	public static <C, R> FixFn<C, R, RuntimeException> fix(final R value) {
		return new FixFn<C, R, RuntimeException>(value);
	}
	
	public static <C, R, E extends Exception> IfFn<C, R, E> ifTrue(
			final Fn<? super C, ?, ? extends E> ifFn,
			final Fn<? super C, ? extends R, ? extends E> trueFn,
			final Fn<? super C, ? extends R, ? extends E> falseFn) {
		return new IfFn<C, R, E>(ifFn, trueFn, falseFn);
	}
	
	public static <A, B, C, E extends Exception> JoinFn<A, B, C, E> join(
			final Fn<? super A, ? extends B, ? extends E> fn0,
			final Fn<? super B, ? extends C, ? extends E> fn1) {
		return new JoinFn<A, B, C, E>(fn0, fn1);
	}

	public static <A, B, C, D, E extends Exception> Fn<A, D, E> join(
			final Fn<? super A, ? extends B, ? extends E> fn0,
			final Fn<? super B, ? extends C, ? extends E> fn1,
			final Fn<? super C, ? extends D, ? extends E> fn2) {
		return join(join(fn0, fn1), fn2);
	}

	public static <A, B, C, D, E, T extends Exception> Fn<A, E, T> join(
			final Fn<? super A, ? extends B, ? extends T> fn0,
			final Fn<? super B, ? extends C, ? extends T> fn1,
			final Fn<? super C, ? extends D, ? extends T> fn2,
			final Fn<? super D, ? extends E, ? extends T> fn3) {
		return join(join(fn0, fn1), join(fn2, fn3));
	}

	public static <A, B, C, D, E, F, T extends Exception> Fn<A, F, T> join(
			final Fn<? super A, ? extends B, ? extends T> fn0,
			final Fn<? super B, ? extends C, ? extends T> fn1,
			final Fn<? super C, ? extends D, ? extends T> fn2,
			final Fn<? super D, ? extends E, ? extends T> fn3,
			final Fn<? super E, ? extends F, ? extends T> fn4) {
		return join(join(fn0, fn1, fn2), join(fn3, fn4));
	}
	
	public static <C, R, E extends Exception> RecodeFn<C, R, E> recode(
			final Fn<? super C, ? extends R, ? extends E> fn,
			final String name,
			final Collection<String> nameList,
			final Collection<? super C> contextList,
			final Collection<? super R> returnList) {
		return new RecodeFn<C, R, E>(fn, name, nameList, contextList, returnList);
	}
	
	public static <C, R, E extends Exception> RecodeFn<C, R, E> recode(
			final Fn<? super C, ? extends R, ? extends E> fn,
			final String name,
			final Collection<String> nameList) {
		return new RecodeFn<C, R, E>(fn, name, nameList, null, null);
	}
	
	public static <C, R, E extends Exception> RecodeFn<C, R, E> recode(
			final Fn<? super C, ? extends R, ? extends E> fn,
			final Collection<? super C> contextList,
			final Collection<? super R> returnList) {
		return new RecodeFn<C, R, E>(fn, null, null, contextList, returnList);
	}
	
	public static <C, R, E extends Exception> SeqFn<C, R, E> seq(
			final Iterable<? extends Fn<? super C, ? extends R, ? extends E>> iterable) {
		return new SeqFn<C, R, E>(iterable);
	}
	
	public static <C, R, E extends Exception> SwitchFn<C, R, E> switching(
			final Map<? super C, ? extends R> map,
			final Fn<? super C, ? extends R, ? extends E> defaultFn) {
		return new SwitchFn<C, R, E>(map, defaultFn);
	}
	
	/**
	 * 
	 * @param <C>
	 * @param <R>
	 * @param map
	 * @param defaultValue 引数に対応する値が null の場合の値。
	 * @return 関数。
	 */
	public static <C, R> SwitchFn<C, R, RuntimeException> switching(
			final Map<? super C, ? extends R> map,
			final R defaultValue) {
		return switching(map, FnUtils.<C, R>fix(defaultValue));
	}
	
	/**
	 * 引数に対応する値が null の場合はnullを返します。
	 * @param <C>
	 * @param <R>
	 * @param map
	 * @return 関数。
	 */
	public static <C, R> SwitchFn<C, R, RuntimeException> switching(
			final Map<? super C, ? extends R> map) {
		return switching(map, FnUtils.<C, R>fix(null));
	}
	
	public static <C, R, E extends Exception> ThrowFn<C, R, E> throwing(final E exception) {
		return new ThrowFn<C, R, E>(exception);
	}
	
	public static <C, R, E extends Exception> TryFn<C, R, E> trying(
			final Fn<? super C, ? extends R, ? extends Exception> fn,
			final Fn<? super Throwable, ? extends R, ? extends E> reportFn,
			final Fn<? super C, ?, ? extends E> finallyFn) {
		return new TryFn<C, R, E>(fn, reportFn, finallyFn);
	}
	
	public static <C, R, E extends Exception> Delegator<C, R, E> delegate(
			final Fn<? super C, ? extends R, ? extends E> fn) {
		return new Delegator<C, R, E>(fn);
	}
	
	public static <C, R, E extends Exception> Delegator<C, R, E> delegate() {
		return new Delegator<C, R, E>(null);
	}
}
