/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import junit.framework.TestCase;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

import woolpack.adapter.JXP;
import woolpack.convert.ConvertUtils;
import woolpack.el.PathEL;
import woolpack.factory.FactoryUtils;
import woolpack.fn.FixFn;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.utils.Utils;

public class XmlUtilsTest extends TestCase {
	
	public void testNodeEL() {
		final Node node = XmlTransformerUtils.convert("<HTML><BODY></BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		XmlUtils.NODE_EL.setValue(context, node);
		assertSame(node, XmlUtils.NODE_EL.getValue(context));
	}
	
	public void testBranchByAttrValue() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("//A"), 
				XmlUtils.branchByAttrValue(Arrays.asList("href"), FnUtils.switching(
						Utils
						.map("page0.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page0.do")))
						.map("page1.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page1.do")))
						.map("page2.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page2.do"))),
						FnUtils.fix(XmlUtils.updateAttrValue("href", FnUtils.fix("pagee.do")))
						)));
		
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.html\" >page0</A>"
				+ "<A href=\"page1.html\" >page1</A>"
				+ "<A href=\"page2.html\" >page2</A>"
				+ "<A href=\"page3.html\" >page3</A>"
				+ "</BODY></HTML>"));
		expression.exec(context);

		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.do\" >page0</A>"
				+ "<A href=\"page1.do\" >page1</A>"
				+ "<A href=\"page2.do\" >page2</A>"
				+ "<A href=\"pagee.do\" >page3</A>"
				+ "</BODY></HTML>"), context.getNode()));
	}
	
	public void testCloneNode() {
		final NodeContext context = new SimpleNodeContext();
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "a"
				+ "</BODY></HTML>");
		context.setNode(node);
		XmlUtils.CLONE_NODE.exec(context);
		assertNotSame(node, context.getNode());
		assertTrue(XmlUtils.equalsNode(node, context.getNode()));
	}

	public void testBranchByAttrValueNotOther() {
		final Fn<NodeContext, Void> expression = XmlUtils.findNode(
				new JXP("//A"), 
				XmlUtils.branchByAttrValue(Arrays.asList("href"), FnUtils.switching(
						Utils
						.map("page0.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page0.do")))
						.map("page1.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page1.do")))
						.map("page2.html", XmlUtils.updateAttrValue("href", FnUtils.fix("page2.do")))
						)));
		
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.html\" >page0</A>"
				+ "<A href=\"page1.html\" >page1</A>"
				+ "<A href=\"page2.html\" >page2</A>"
				+ "<A href=\"page3.html\" >page3</A>"
				+ "</BODY></HTML>"));
		expression.exec(context);

		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<A href=\"page0.do\" >page0</A>"
				+ "<A href=\"page1.do\" >page1</A>"
				+ "<A href=\"page2.do\" >page2</A>"
				+ "<A href=\"page3.html\" >page3</A>"
				+ "</BODY></HTML>"), context.getNode()));
	}
	
	public void testFindNode() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<span id=\"id0\" />"
				+ "<span id=\"id0\" />"
				+ "<span id=\"id0\" />"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node);
		XmlUtils.findNode(
				new JXP("id", "id0"),
				XmlUtils.replaceTextToChild(FnUtils.fix("value0")),
				XmlUtils.replaceTextToChild(FnUtils.fix("value1")))
		.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<span id=\"id0\" >value0</span>"
				+ "<span id=\"id0\" >value1</span>"
				+ "<span id=\"id0\" >value1</span>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testGetAttrValue() {
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"));
		context.setNode(context.getNode().getFirstChild().getFirstChild().getFirstChild());
		assertEquals("hoge", XmlUtils.getAttrValue("id").exec(context));
	}
	
	public void testNodeName() {
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"));
		context.setNode(context.getNode().getFirstChild().getFirstChild().getFirstChild());
		assertEquals("div", FnUtils.join(XmlUtils.GET_NODE, XmlUtils.GET_NODE_NAME).exec(context));
	}
	
	public void testInsertElementToChild() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.insertElementToChild("FORM", XmlUtils.updateAttrValue("action", FnUtils.fix("action0")))
		.exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" ><FORM action=\"action0\">"
				+ "hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</FORM></SPAN>"
				+ "</BODY></HTML>"), node));
	}

	public void testInsertElementToChildNotExpression() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.insertElementToChild("FORM").exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" ><FORM>"
				+ "hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</FORM></SPAN>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testInsertElementToParent() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		XmlUtils.insertElementToParent("FORM", XmlUtils.updateAttrValue("action", FnUtils.fix("action0")))
		.exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<FORM action=\"action0\"><SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN></FORM>piyo0</SPAN>"
				+ "</BODY></HTML>"), node));
	}

	public void testInsertElementToParentNotExpression() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		XmlUtils.insertElementToParent("FORM").exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN id=\"id0\" >hoge0<FORM><SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN></FORM>piyo0</SPAN>"
				+ "</BODY></HTML>"), node));
	}

	
	private void scenarioLoop(final Object collection) {
		final Fn<NodeContext, Void> expression = XmlUtils.copyTemplate(
				new PathEL("map.list0"),
				new PathEL("map.bean"),
				XmlUtils.findNode(
						new JXP("id", "name0"),
						XmlUtils.replaceTextToChild(
								FnUtils.join(
										FactoryUtils.doEL(new PathEL("map.bean")),
										ConvertUtils.TO_STRING))));

		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">dummy</SPAN></TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>");
		final MapNodeContext context = new MapNodeContext();
		context.setMap(Utils.map("list0", collection));
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getFirstChild());
		expression.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TABLE>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">value0</SPAN></TD></TR>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">value1</SPAN></TD></TR>"
				+ "<TR id=\"iterate0\"><TD><SPAN id=\"name0\">value2</SPAN></TD></TR>"
				+ "</TABLE>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testLoopList() {
		scenarioLoop(Arrays.asList("value0", "value1", "value2"));
	}
	
	public void testLoopArray() {
		scenarioLoop(new String[]{"value0", "value1", "value2"});
	}

	public void testNodeFactory() {
		final Node node = XmlUtils.nodeFactory(
				FactoryUtils.stringReaderFactory("<HTML><BODY>myxml</BODY></HTML>"),
				XmlTransformerUtils.TRANSFORMER
		).exec("a");
		assertEquals("myxml", node.getFirstChild().getFirstChild().getFirstChild().getNodeValue());
	}

	public void testNormalize() {
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert("<HTML><BODY>a</BODY></HTML>"));
		final Node n1 = new JXP("/HTML/BODY").evaluateOne(context.getNode());
		n1.appendChild(((Document) context.getNode()).createTextNode("b"));
		n1.appendChild(((Document) context.getNode()).createTextNode("c"));
		assertEquals(3, n1.getChildNodes().getLength());
		XmlUtils.NORMALIZE_NODE.exec(context);
		assertEquals(1, n1.getChildNodes().getLength());
	}
	
	public void testRemoveAttr() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.removeAttr("id").exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}

	public void testRemove() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.REMOVE_THIS.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}

	public void testRemoveChildren() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.REMOVE_CHILDREN.exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div></div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testReplaceText() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hoge<TEXTAREA name=\"name0\" >value0</TEXTAREA>piyo"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		XmlUtils.replaceText(FnUtils.fix("value1")).exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hogevalue1piyo"
				+ "</BODY></HTML>"), node));
	}

	public void testReplaceTextNull() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hoge<TEXTAREA name=\"name0\" >value0</TEXTAREA>piyo"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild().getNextSibling());
		XmlUtils.replaceText(FnUtils.fix((String) null)).exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "hoge<TEXTAREA name=\"name0\" >value0</TEXTAREA>piyo"
				+ "</BODY></HTML>"), node));
	}

	public void testReplaceTextToChild() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.replaceTextToChild(FnUtils.fix("value1")).exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value1</TEXTAREA>"
				+ "</BODY></HTML>"), node));
	}

	public void testReplaceTextToChildNull() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.replaceTextToChild(FnUtils.fix((String) null)).exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>"
				+ "</BODY></HTML>"), node));
	}

	public void testRetainChildren() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div>0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.RETAIN_CHILDREN.exec(context);
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "0"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}

	public void testSeekNode() {
		final List<String> list = new ArrayList<String>();
		final Fn<NodeContext, Void> expression = 
				XmlUtils.seekNode(new Fn<NodeContext, Void>() {
					public Void exec(final NodeContext context) {
						list.add(context.getNode().getNodeName());
						return null;
					}
				});
		
		final NodeContext context = new SimpleNodeContext();
		context.setNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<SPAN><SPAN>a</SPAN><SPAN>b</SPAN></SPAN><SPAN>c</SPAN>"
				+ "</BODY></HTML>"));
		expression.exec(context);

		final List<String> expected = new ArrayList<String>();
		expected.add("#document");
		expected.add("HTML");
		expected.add("BODY");
		expected.add("SPAN");
		expected.add("SPAN");
		expected.add("#text");
		expected.add("SPAN");
		expected.add("#text");
		expected.add("SPAN");
		expected.add("#text");
		assertEquals(expected, list);
	}
	
	public void testSetNode() {
		final Node node = XmlTransformerUtils.convert("<HTML><BODY></BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		XmlUtils.setNode(new FixFn<NodeContext, Node>(node)).exec(context);
		assertSame(node, context.getNode());
	}

	public void testUpdateAttrValue() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.updateAttrValue("id", FnUtils.fix("piyo")).exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"piyo\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}
	
	public void testUpdateAttrValueNull() {
		final Node node = XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>");
		final NodeContext context = new SimpleNodeContext();
		context.setNode(node.getFirstChild().getFirstChild().getFirstChild());
		XmlUtils.updateAttrValue("id", FnUtils.fix((String) null)).exec(context);
		node.normalize();
		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert(
				"<HTML><BODY>"
				+ "<div id=\"hoge\">0</div>"
				+ "<span>1</span>"
				+ "</BODY></HTML>"), node));
	}


	private void scenarioEqualsNode(final boolean expected, final String s0, final String s1) {
		final Node node0 = XmlTransformerUtils.convert("<HTML><BODY>" + s0 + "</BODY></HTML>");
		final Node node1 = XmlTransformerUtils.convert("<HTML><BODY>" + s1 + "</BODY></HTML>");
		assertEquals(expected, XmlUtils.equalsNode(node0, node1));
		assertTrue(node0 != node1);
	}

	public void testDefferentDocument() {
		scenarioEqualsNode(true,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>");
	}

	public void testAttrOrder() {
		scenarioEqualsNode(true,
				"<!-- hoge -->" + "<TEXTAREA id=\"id0\" name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" id=\"id0\" >value0</TEXTAREA>");
	}

	public void testDefferentAttrValue() {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name1\" >value0</TEXTAREA>");
	}

	public void testDefferentAttrRack() {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA >value0</TEXTAREA>");
	}

	public void testDefferentOption() {
		scenarioEqualsNode(true,
				"<!-- hoge -->" + "<SELECT name=\"name0\" ><OPTION value=\"value0\" >label0</OPTION></SELECT>",
				"<!-- hoge -->" + "<SELECT name=\"name0\" ><OPTION value=\"value0\" >label0</OPTION></SELECT>");
	}

	public void testDefferentChildRack() {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<SELECT name=\"name0\" ><OPTION value=\"value0\" >label0</OPTION></SELECT>",
				"<!-- hoge -->" + "<SELECT name=\"name0\" ></SELECT>");
	}

	public void testDefferentAttrName() {
		scenarioEqualsNode(false, "<!-- hoge -->"
				+ "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA namf=\"name0\" >value0</TEXTAREA>");
	}

	public void testDefferentText() {
		scenarioEqualsNode(false, 
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value1</TEXTAREA>");
	}

	public void testDefferentComment() {
		scenarioEqualsNode(false,
				"<!-- hoge -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>",
				"<!-- piyo -->" + "<TEXTAREA name=\"name0\" >value0</TEXTAREA>");
	}

	public void testNull0() {
		final Node node = XmlTransformerUtils.convert("<!-- hoge --><TEXTAREA name=\"name0\" >value0</TEXTAREA>");
		assertTrue(XmlUtils.equalsNode(null, null));
		assertFalse(XmlUtils.equalsNode(null, node));
		assertFalse(XmlUtils.equalsNode(node, null));
	}

	public void testDefferentElementName() {
		scenarioEqualsNode(false, 
				"<!-- hoge -->" + "<SPAN>hoge</SPAN>",
				"<!-- hoge -->" + "<DIV>hoge</DIV>");
	}

	public void testDefferentNodeType() {
		scenarioEqualsNode(false, 
				"<!-- hoge -->" + "hoge",
				"<!-- hoge -->" + "<DIV>hoge</DIV>");
	}
}
