/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import woolpack.fn.Fn;

/**
 * 現在の行を{@link Map}にコピーして返す{@link Fn}です。
 * @author nakamura
 *
 */
public class MapResult implements Fn<ResultSet, Map<String, Object>> {
	private static final Fn<? super Integer, ? extends Map<String, Object>> DEFAULT_MAP_FACTORY = new Fn<Integer, Map<String, Object>>() {
		public Map<String, Object> exec(final Integer c) {
			return new HashMap<String, Object>(c);
		}
	};
	
	private Fn<? super Integer, ? extends Map<String, Object>> mapFactory;
	
	public MapResult(final Fn<? super Integer, ? extends Map<String, Object>> mapFactory) {
		this.mapFactory = mapFactory;
	}
	
	public MapResult() {
		this(DEFAULT_MAP_FACTORY);
	}
	
	public Map<String, Object> exec(final ResultSet c) {
		try {
			final ResultSetMetaData metaData = c.getMetaData();
			final int size = metaData.getColumnCount();
			final Map<String, Object> map = mapFactory.exec(size);
			for (int i = 0; i < size; i++) {
				map.put(metaData.getColumnName(i + 1), c.getObject(i + 1));
			}
			return map;
		} catch (final SQLException e) {
			throw new IllegalStateException(e);
		}
	}

	public Fn<? super Integer, ? extends Map<String, Object>> getMapFactory() {
		return mapFactory;
	}
	public void setMapFactory(
			final Fn<? super Integer, ? extends Map<String, Object>> mapFactory) {
		this.mapFactory = mapFactory;
	}
}
