/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Map;

import javax.sql.DataSource;

import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 委譲先から{@link PreparedStatementInfo}を取得し、
 * 引数を{@link PreparedStatement}に設定して
 * {@link PreparedStatement#execute()}を実行し、
 * 返却値の生成を委譲する{@link Fn}です。
 * @author nakamura
 *
 * @param <R>
 */
public class MapInput<R> implements Fn<Map<String, Object>, R> {
	private DataSource dataSource;
	private Fn<? super Map<String, Object>, ? extends PreparedStatementInfo> queryFactory;
	private Fn<? super PreparedStatement, ? extends R> converter;
	
	/**
	 * 
	 * @param dataSource データソース。
	 * @param queryFactory クエリ情報生成の委譲先。
	 * @param converter 変換の委譲先。
	 */
	public MapInput(
			final DataSource dataSource,
			final Fn<? super Map<String, Object>, ? extends PreparedStatementInfo> queryFactory,
			final Fn<? super PreparedStatement, ? extends R> converter) {
		this.dataSource = dataSource;
		this.queryFactory = queryFactory;
		this.converter = converter;
	}
	
	/**
	 * 
	 * @param dataSource データソース。
	 * @param info クエリ情報。
	 * @param converter 変換の委譲先。
	 */
	public MapInput(
			final DataSource dataSource,
			final PreparedStatementInfo info,
			final Fn<? super PreparedStatement, ? extends R> converter) {
		this(dataSource, FnUtils.fix(info), converter);
	}

	public R exec(final Map<String, Object> c) {
		try {
			final PreparedStatementInfo info = queryFactory.exec(c);
			final Connection connection = dataSource.getConnection();
			try {
				final PreparedStatement statement = connection.prepareStatement(info.getQuery());
				try {
					for (int i = 0; i < info.getList().size(); i++) {
						statement.setObject(i + 1, c.get(info.getList().get(i)));
					}
					statement.execute();
					return converter.exec(statement);
				} finally {
					statement.close();
				}
			} finally {
				connection.close();
			}
		} catch (final SQLException e) {
			throw new IllegalStateException(e);
		}
	}

	public DataSource getDataSource() {
		return dataSource;
	}
	public void setDataSource(final DataSource dataSource) {
		this.dataSource = dataSource;
	}
	public Fn<? super Map<String, Object>, ? extends PreparedStatementInfo> getQueryFactory() {
		return queryFactory;
	}
	public void setQueryFactory(final Fn<? super Map<String, Object>, ? extends PreparedStatementInfo> queryFactory) {
		this.queryFactory = queryFactory;
	}
	public Fn<? super PreparedStatement, ? extends R> getConverter() {
		return converter;
	}
	public void setConverter(final Fn<? super PreparedStatement, ? extends R> converter) {
		this.converter = converter;
	}
}
