/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import woolpack.fn.Fn;

/**
 * {@link Statement#getResultSet()}の{@link ResultSet#next()}でカーソルを移動しながら
 * 委譲先にレコード情報の生成を委譲して、
 * その結果を{@link List}に格納して返す{@link Fn}です。
 * @author nakamura
 *
 * @param <R>
 */
public class ListResult<R> implements Fn<Statement, List<R>> {
	private Fn<? super ResultSet, ? extends R> rowFn;
	private int max;
	
	/**
	 * 
	 * @param fn 委譲先。
	 * @param max 最大この数まで{@link ResultSet#next()}を呼び出す閾値。
	 */
	public ListResult(final Fn<? super ResultSet, ? extends R> fn, final int max) {
		this.rowFn = fn;
		this.max = max;
	}
	
	/**
	 * 最大{@link Integer#MAX_VALUE}回{@link ResultSet#next()}を呼び出します。
	 * @param fn 委譲先。
	 */
	public ListResult(final Fn<? super ResultSet, ? extends R> fn) {
		this(fn, Integer.MAX_VALUE);
	}

	public List<R> exec(final Statement c) {
		try {
			final ResultSet resultSet = c.getResultSet();
			try {
				final List<R> list = new ArrayList<R>();
				int i = 0;
				while (resultSet.next()) {
					i++;
					if (i > max) {
						break;
					}
					list.add(rowFn.exec(resultSet));
				}
				return list;
			} finally {
				resultSet.close();
			}
		} catch (final SQLException e) {
			throw new IllegalStateException(e);
		}
	}

	public Fn<? super ResultSet, ? extends R> getRowFn() {
		return rowFn;
	}
	public void setRowFn(Fn<? super ResultSet, ? extends R> fn) {
		this.rowFn = fn;
	}
	public int getMax() {
		return max;
	}
	public void setMax(final int max) {
		this.max = max;
	}
}
