/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;

import woolpack.el.ELTargetRuntimeException;
import woolpack.el.PropertyEL;
import woolpack.fn.Fn;

/**
 * アクションを実行するクラスです。
 * {@link #invoke(Object, String, Map)}メソッドは、
 * アクションidに対するアクション定義を検索し、
 * root から{@link ActionDef#getComponentEL()}でコンポーネントを取得し、
 * inputMap の値をコンポーネントに設定し、
 * root のメソッド{@link ActionDef#getMethodEL()}を実行し、
 * 実行結果にマッチする遷移先定義を{@link ActionDef#getForwardDefIterable()}から検索します。
 * アクション定義に遷移先定義が見つからない場合、
 * コンストラクタ引数の{@link ForwardDef}の一覧から遷移先定義を検索します。
 * 
 * <br/>適用しているデザインパターン：Template Method。
 * 
 * @author nakamura
 * 
 */
public class ActionInvoker {
	private Fn<String, ActionDef> switching;
	private Iterable<ForwardDef> iterable;

	/**
	 * @param switching アクション id とアクション定義の対応表。
	 * @param iterable 遷移先定義の一覧。
	 */
	public ActionInvoker(
			final Fn<String, ActionDef> switching,
			final Iterable<ForwardDef> iterable) {
		this.switching = switching;
		this.iterable = iterable;
	}

	/**
	 * @param switching アクション id とアクション定義の対応表。
	 * @param array 遷移先定義の一覧。
	 */
	public ActionInvoker(
			final Fn<String, ActionDef> switching,
			final ForwardDef... array) {
		this(switching, Arrays.asList(array));
	}

	/**
	 * アクションを実行します。
	 * 
	 * @param root コンポーネントとメソッド検索の基点。
	 * @param id アクション id。
	 * @param inputMap
	 *            入力値のMap。{@link javax.servlet.ServletRequest#getParameterMap()}
	 *            またはそれを
	 *            {@link woolpack.validator.ValidatorUtils#convert(Map)}
	 *            で変換した結果が渡されることを想定しています。
	 * @return 実行結果。
	 * @throws ForwardDefNotFoundRuntimeException 該当する遷移先定義が見つからない場合。
	 */
	public ActionResult invoke(
			final Object root,
			final String id,
			final Map inputMap) {
		final ActionDef actionDef = switching.exec(id);
		setValuesTo(actionDef.getComponentEL().getValue(root), inputMap, this);
		try {
			try {
				final Object result = actionDef.getMethodEL().getValue(root);
				return findId(actionDef.getForwardDefIterable(), result);
			} catch (final ELTargetRuntimeException e) {
				return findId(actionDef.getForwardDefIterable(), e.getCause());
			}
		} catch (final ForwardDefNotFoundRuntimeException e) {
			return findId(iterable, e.getReturnedObject());
		}
	}

	/**
	 * inputMap の値を root に設定します。
	 * 
	 * @param root 基点。null なら設定しません。
	 * @param inputMap 設定する値の{@link Map}。
	 * @throws NullPointerException inputMap が null の場合。
	 */
	public static void setValuesTo(
			final Object root,
			final Map inputMap,
			final ActionInvoker invoker) {
		if (root == null) {
			return;
		}
		for (final Object entryObject : inputMap.entrySet()) {
			final Entry entry = (Entry) entryObject;
			try {
				new PropertyEL((String) entry.getKey())
				.setValue(root, entry.getValue());
			} catch (final RuntimeException exception) {
				if (invoker != null) {
					invoker.handleRuntimeException(exception);
				}
			}
		}
	}
	
	/**
	 * {@link #setValuesTo(Object, Map, ActionInvoker)}が失敗した際に呼び出されます(called)。
	 * デフォルトは何もしません。
	 * @param exception {@link #setValuesTo(Object, Map, ActionInvoker)}が失敗した際のエラーオブジェクト。
	 */
	protected void handleRuntimeException(final RuntimeException exception) {
		// do nothing.
	}

	/**
	 * 遷移先定義を検索して実行結果を返します。
	 * 
	 * @param iterable 遷移先定義の一覧。
	 * @param aReturnedObject メソッドが返却したオブジェクト。
	 * @return 実行結果。
	 * @throws ForwardDefNotFoundRuntimeException 該当する遷移先定義が見つからない場合。
	 */
	public static ActionResult findId(final Iterable<ForwardDef> iterable,
			final Object aReturnedObject) {
		for (final ForwardDef forwardDef : iterable) {
			if (forwardDef.getMatcher().exec(aReturnedObject)) {
				return new ActionResult(forwardDef, aReturnedObject);
			}
		}
		throw new ForwardDefNotFoundRuntimeException(aReturnedObject);
	}

	public Iterable<ForwardDef> getIterable() {
		return iterable;
	}
	public void setIterable(final Iterable<ForwardDef> iterable) {
		this.iterable = iterable;
	}
	public Fn<String, ActionDef> getSwitching() {
		return switching;
	}
	public void setSwitching(final Fn<String, ActionDef> switching) {
		this.switching = switching;
	}
}
