/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.frame;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.regex.Pattern;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Node;

import woolpack.adapter.JXPFactory;
import woolpack.convert.ConvertUtils;
import woolpack.ee.EEContext;
import woolpack.ee.FrameToTable;
import woolpack.ee.ServletInputStreamFactory;
import woolpack.factory.FactoryUtils;
import woolpack.fn.Delegator;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.html.HtmlUtils;
import woolpack.id.IdConverter;
import woolpack.id.IdUtils;
import woolpack.utils.BuildableArrayList;
import woolpack.xml.NodeSetter;
import woolpack.xml.XmlTransformerContext;
import woolpack.xml.XmlUtils;

/**
 * FRAME を TABLE に変換するサンプル Web アプリケーションです。
 * @author nakamura
 *
 */
public class FrameServlet extends HttpServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public final transient Delegator<String, Node> toNode;
	public final transient Fn<EEContext, Void> fn;
	
	public FrameServlet() {
		super();
		toNode = new Delegator<String, Node>();

		fn = FnUtils.seq(new BuildableArrayList<Fn<? super EEContext, Void>>()
				.list(new IdConverter<Void>(ConvertUtils.convertRegExp(Pattern.compile("^.*/([^/]+)$"), "$1")))
				.list(new FrameToTable("common_frame", "target", FnUtils.seq(new BuildableArrayList<Fn<? super EEContext, Void>>()
						.list(new NodeSetter<EEContext>(FnUtils.join(IdUtils.GET_ID, toNode)))
						.list(HtmlUtils.NORMALIZE_CASE)
						.list(HtmlUtils.removeExtension("A", "href"))
						.list(HtmlUtils.removeExtension("FORM", "action"))
						.list(HtmlUtils.removeExtension("FRAME", "src"))
				), new JXPFactory()))
		);
	}

	@Override public void init(final ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		toNode.setFn(FnUtils.join(
				ConvertUtils.convertRegExp(Pattern.compile("^(.*)$"), "/html/sample/frame/$1.html"),
				XmlUtils.nodeFactory(
						FactoryUtils.inputStreamReaderFactory(new ServletInputStreamFactory(
						servletConfig.getServletContext()), "UTF-8"),
						XmlUtils.TRANSFORMER)));
	}

	@Override protected void service(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
		final EEContext context = new EEContext();
		context.setId(request.getRequestURI());
		fn.exec(context);
		final Writer w = new BufferedWriter(new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
		try {
			final XmlTransformerContext tc = new XmlTransformerContext();
			tc.setSource(new DOMSource(context.getNode()));
			tc.setResult(new StreamResult(w));
			XmlUtils.TRANSFORMER.exec(tc);
		} finally {
			w.close();
		}
	}
}
