/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import woolpack.bool.BooleanOperator;
import woolpack.fn.Fn;
import woolpack.utils.Utils;

/**
 * ユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 * 
 */
public final class ValidatorUtils {
	public static final ValueGetter VALUE = new ValueGetter();
	public static final MapGetter MAP = new MapGetter();
	public static final ValuesGetter VALUES = new ValuesGetter();

	private ValidatorUtils() {
	}

	/**
	 * 引数のキーが全て文字列型・値が全てオブジェクトの一覧とみなして変換します。
	 * サーブレットAPIの request.getParameterMap() を変換するために定義しています。
	 * 返却値を更新しても引数には影響しません。
	 * @param map
	 * @return 変換結果。
	 */
	public static Map<String, List<Object>> convert(final Map map) {
		final Map<String, List<Object>> map1 = new HashMap<String, List<Object>>();
		for (final Object entryObject : map.entrySet()) {
			final Entry entry = (Entry) entryObject;
			final Iterable c = Utils.toIterable(entry.getValue());
			final List<Object> list = new ArrayList<Object>();
			for (final Object o : c) {
				list.add(o);
			}
			map1.put((String) entry.getKey(), list);
		}
		return map1;
	}
	
	public static NameBranchIfExists branchByNameIfExists(
			final BooleanOperator<ValidatorContext> operator,
			final Map<String, ? extends Fn<ValidatorContext, Boolean>> map) {
		return new NameBranchIfExists(operator, map);
	}
	
	public static NameBranchIfExists branchByNameIfExists(
			final Map<String, ? extends Fn<ValidatorContext, Boolean>> map) {
		return new NameBranchIfExists(map);
	}
	
	public static NameBranch branchByName(
			final BooleanOperator<ValidatorContext> operator,
			final Map<String, ? extends Fn<ValidatorContext, Boolean>> map) {
		return new NameBranch(operator, map);
	}
	
	public static NameBranch branchByName(
			final Map<String, ? extends Fn<ValidatorContext, Boolean>> map) {
		return new NameBranch(map);
	}
	
	public static ConvertValidator convertValue(final Fn<Object, ?> fn) {
		return new ConvertValidator(fn);
	}
	
	public static MessageValidator message(final String message) {
		return new MessageValidator(message);
	}
	
	public static LocalIndexValidator localIndex(
			final int tmpIndex,
			final Fn<ValidatorContext, Boolean> fn) {
		return new LocalIndexValidator(tmpIndex, fn);
	}
	
	public static LocalKeyValidator localKey(
			final String tmpKey,
			final Fn<ValidatorContext, Boolean> fn) {
		return new LocalKeyValidator(tmpKey, fn);
	}
	
	public static ValueLoopValidator loopValue(
			final BooleanOperator<ValidatorContext> operator,
			final Fn<? super ValidatorContext, Boolean> fn) {
		return new ValueLoopValidator(operator, fn);
	}
	
	public static ValueLoopValidator loopValue(
			final Fn<? super ValidatorContext, Boolean> fn) {
		return new ValueLoopValidator(fn);
	}
}
