/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.LinkedHashMap;

import junit.framework.TestCase;

import org.w3c.dom.Element;

import woolpack.adapter.JXP;
import woolpack.adapter.JXPFactory;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Serial;
import woolpack.dom.XPath;
import woolpack.el.AbstractToELTargetExceptionEL;
import woolpack.el.ELConstants;
import woolpack.el.PathEL;
import woolpack.test.TestUtils;
import woolpack.utils.MapBuilder;

public class MakeSelectTest extends TestCase {

	public void testConstructor() {
		try {
			new MakeSelect(null, new JXPFactory());
			fail();
		} catch (final NullPointerException expected) {
		}
		try {
			new MakeSelect(ELConstants.NULL, null);
			fail();
		} catch (final NullPointerException expected) {
		}
	}

	public void testSelectChild() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeForm("<SELECT name=\"name1\" >"
						+ "<OPTION value=\"value0\">label0</OPTION>"
						+ "<OPTION value=\"value1\" />label1"
						+ "<OPTION value=\"value2\">label2</OPTION>"
						+ "</SELECT>"), new XPath(new JXP("name", "name1"),
				new MakeSelect(new PathEL("local.map0"), new JXPFactory())));
		final DomContext context = new DomContext();
		context.getLocal().put(
				"map0",
				MapBuilder.get(new LinkedHashMap<String, Object>()).put("v0",
						"l0").put("v1", "l1").get());

		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<SELECT name=\"name1\" >"
				+ "<OPTION value=\"v0\">l0</OPTION>"
				+ "<OPTION value=\"v1\">l1</OPTION>" + "</SELECT>"));
	}

	public void testNotFound() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeForm("<SELECT name=\"name1\" >"
						+ "<OPTION value=\"value0\">label0</OPTION>"
						+ "<OPTION value=\"value1\" />label1"
						+ "<OPTION value=\"value2\">label2</OPTION>"
						+ "</SELECT>"), new XPath(new JXP("name", "name1"),
				new MakeSelect(new PathEL("local.map0"), new JXPFactory())));
		final DomContext context = new DomContext();

		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<SELECT name=\"name1\" >"
				+ "<OPTION value=\"value0\">label0</OPTION>"
				+ "<OPTION value=\"value1\" />label1"
				+ "<OPTION value=\"value2\">label2</OPTION>" + "</SELECT>"));
	}

	public void testSelectSibling() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeForm("<SELECT name=\"name1\" >"
						+ "<OPTION value=\"value0\" />label0"
						+ "<OPTION value=\"value1\">label1</OPTION>"
						+ "<OPTION value=\"value2\" />label2" + "</SELECT>"),
				new XPath(new JXP("name", "name1"), new MakeSelect(new PathEL(
						"local.map0"), new JXPFactory())));
		final DomContext context = new DomContext();
		context.getLocal().put(
				"map0",
				MapBuilder.get(new LinkedHashMap<String, Object>()).put("v0",
						"l0").put("v1", "l1").get());

		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<SELECT name=\"name1\" >"
				+ "<OPTION value=\"v0\">l0</OPTION>"
				+ "<OPTION value=\"v1\">l1</OPTION>" + "</SELECT>"));
	}

	public void testTipsAuto() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeForm("<SELECT name=\"name1\" >"
						+ "<OPTION value=\"value0\">label0</OPTION>"
						+ "<OPTION value=\"value1\">label1</OPTION>"
						+ "<OPTION value=\"value2\">label2</OPTION>"
						+ "</SELECT>"), new XPath(new JXP("//SELECT"),
				new MakeSelect(new AbstractToELTargetExceptionEL() {
					@Override
					public Object execute(final Object root, final Object value)
							throws Exception {
						// new OGE("local[node.getAttribute(\"name\")]")
						final DomContext context = (DomContext) root;
						return context.getLocal().get(
								((Element) context.getNode())
										.getAttribute("name"));
					}
				}, new JXPFactory())));
		final DomContext context = new DomContext();
		context.getLocal().put(
				"name1",
				MapBuilder.get(new LinkedHashMap<String, Object>()).put("v0",
						"l0").put("v1", "l1").get());

		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<SELECT name=\"name1\" >"
				+ "<OPTION value=\"v0\">l0</OPTION>"
				+ "<OPTION value=\"v1\">l1</OPTION>" + "</SELECT>"));
	}
}
