/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;

import junit.framework.TestCase;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.test.TestUtils;
import woolpack.test.ee.EETestUtils;
import woolpack.utils.MapBuilder;

public class SessionClearBuilderTest extends TestCase {

	public void testConstractor() {
		try {
			new SessionClearBuilder(null, "nameKey", "removerKey",
					new HashMap<String, String>(), new HashSet<String>());
			fail();
		} catch (final NullPointerException e) {
		}
		try {
			new SessionClearBuilder("", "nameKey", "removerKey",
					new HashMap<String, String>(), new HashSet<String>());
			fail();
		} catch (final StringIndexOutOfBoundsException e) {
		}
		try {
			new SessionClearBuilder("sessionIdKey", null, "removerKey",
					new HashMap<String, String>(), new HashSet<String>());
			fail();
		} catch (final NullPointerException e) {
		}
		try {
			new SessionClearBuilder("sessionIdKey", "", "removerKey",
					new HashMap<String, String>(), new HashSet<String>());
			fail();
		} catch (final StringIndexOutOfBoundsException e) {
		}
		try {
			new SessionClearBuilder("sessionIdKey", "nameKey", null,
					new HashMap<String, String>(), new HashSet<String>());
			fail();
		} catch (final NullPointerException e) {
		}
		try {
			new SessionClearBuilder("sessionIdKey", "nameKey", "",
					new HashMap<String, String>(), new HashSet<String>());
			fail();
		} catch (final StringIndexOutOfBoundsException e) {
		}
		try {
			new SessionClearBuilder("sessionIdKey", "nameKey", "removerKey",
					null, new HashSet<String>());
			fail();
		} catch (final NullPointerException e) {
		}
		try {
			new SessionClearBuilder("sessionIdKey", "nameKey", "removerKey",
					new HashMap<String, String>(), null);
			fail();
		} catch (final NullPointerException e) {
		}
	}

	public void testNormal() {
		final Map<String, String> map = new HashMap<String, String>();
		final Set<String> set = new HashSet<String>();
		final SessionClearBuilder builder = new SessionClearBuilder(map, set);
		final DomExpression init = builder.getInitExpression();
		final DomExpression clear = builder.getClearExpression();

		final DomContext context0 = new DomContext();
		final ConcurrentMap<String, Object> session0 = EETestUtils
				.generateSession();
		context0.setSession(session0);

		// セッション識別子が設定されていない場合は何もしない
		init.interpret(context0);
		assertTrue(map.isEmpty());
		assertTrue(set.isEmpty());

		// ログイン名が設定されていない場合は何もしない
		session0.put(SessionClearBuilder.SESSION_ID_KEY, "session0");
		init.interpret(context0);
		assertTrue(map.isEmpty());
		assertTrue(set.isEmpty());

		// セッション識別子とログイン名が設定されている場合は map に設定する
		session0.put(SessionClearBuilder.NAME_KEY, "name0");
		init.interpret(context0);
		assertTrue(TestUtils.equals(MapBuilder.get(
				new HashMap<String, Object>()).put("session0", "name0").get(),
				map));
		assertTrue(set.isEmpty());

		// 既に設定されている場合はなにもしない
		init.interpret(context0);
		assertTrue(TestUtils.equals(MapBuilder.get(
				new HashMap<String, Object>()).put("session0", "name0").get(),
				map));
		assertTrue(set.isEmpty());

		// 複数のセッションを管理する
		final DomContext context1 = new DomContext();
		final ConcurrentMap<String, Object> session1 = EETestUtils
				.generateSession();
		context1.setSession(session1);
		session1.put(SessionClearBuilder.SESSION_ID_KEY, "session1");
		session1.put(SessionClearBuilder.NAME_KEY, "name1");
		init.interpret(context1);
		assertTrue(TestUtils.equals(MapBuilder.get(
				new HashMap<String, Object>()).put("session0", "name0").put(
				"session1", "name1").get(), map));
		assertTrue(set.isEmpty());

		// セッションクリア対象を設定
		set.add("session0");

		// セッションクリア対象でない場合はなにもしない
		clear.interpret(context1);
		assertTrue(TestUtils.equals(MapBuilder.get(
				new HashMap<String, Object>()).put("session0", "name0").put(
				"session1", "name1").get(), map));
		assertTrue(TestUtils.equals(Arrays.asList("session0"), set));

		// セッションクリア対象の場合はクリアする
		clear.interpret(context0);
		assertTrue(TestUtils.equals(MapBuilder.get(
				new HashMap<String, Object>()).put("session1", "name1").get(),
				map));
		assertTrue(set.isEmpty());

	}
}
