/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor.impl;

import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import woolpack.utils.MapBuilder;
import woolpack.visitor.Acceptable;
import woolpack.visitor.Visitor;
import woolpack.visitor.VisitorConstants;

/**
 * コンストラクタツリーを日本語 HTML ドキュメントに変換するための定数の集まり。
 * 
 * @author nakamura
 * 
 */
public final class DocBaseConstants {

	/**
	 * 数値変換を省略表記に変換する{@link Acceptable}。
	 */
	public static final Acceptable<Object> ABBR_NUMBER_FORMAT = VisitorConstants
			.visitAcceptable("数値変換フォーマット");

	/**
	 * 日付変換を省略表記に変換する{@link Acceptable}。
	 */
	public static final Acceptable<Object> ABBR_DATE_FORMAT = VisitorConstants
			.visitAcceptable("日付変換フォーマット");

	/**
	 * {@link SimpleDateFormat}で設定可能なパラメータの内、
	 * コンストラクタで設定可能な内容の表記に変換する{@link Acceptable}。
	 */
	public static final Acceptable<SimpleDateFormat>
	CONSTRUCTOR_SIMPLE_DATE_FORMAT
	= new Acceptable<SimpleDateFormat>() {
		public void accept(final Visitor visitor, final SimpleDateFormat v) {
			visitor.visit("パターン\"");
			visitor.visit(v.toPattern());
			visitor.visit("\"の日付フォーマット");
		}
	};

	/**
	 * {@link DecimalFormat}で設定可能なパラメータの内、
	 * コンストラクタで設定可能な内容の表記に変換する{@link Acceptable}。
	 */
	public static final Acceptable<DecimalFormat> CONSTRUCTOR_DECIMAL_FORMAT
	= new Acceptable<DecimalFormat>() {
		public void accept(final Visitor visitor, final DecimalFormat v) {
			visitor.visit("パターン\"");
			visitor.visit(v.toPattern());
			visitor.visit("\"の数値フォーマット");
		}
	};

	/**
	 * 基本 API に関する委譲先{@link Acceptable}を分岐する定義。 コンストラクタツリーを日本語 HTML
	 * ドキュメントに変換するために使用する{@link Map}。
	 */
	public static final Map<Object, Acceptable> BASE = Collections
			.unmodifiableMap(MapBuilder.get(
					new LinkedHashMap<Object, Acceptable>()).put(
					SimpleDateFormat.class, CONSTRUCTOR_SIMPLE_DATE_FORMAT).put(
					DecimalFormat.class, CONSTRUCTOR_DECIMAL_FORMAT).put(
					DateFormat.class, ABBR_DATE_FORMAT).put(NumberFormat.class,
					ABBR_NUMBER_FORMAT).get());

	private DocBaseConstants() {
	} // カバレージがここを通過してはいけない
}
