/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.text.Format;
import java.text.ParseException;

import woolpack.text.CloneFormatFactory;
import woolpack.text.FormatFactory;
import woolpack.utils.CheckUtils;

/**
 * {@link Format#parseObject(java.lang.String)}を使用して
 * {@link ValidatorContext#getTmpValue()}
 * を変換してその成否を返す{@link ValidatorExpression}。
 * 適用しているパターン：Adapter。
 * 
 * @author nakamura
 * 
 */
public class ParseValidator implements ValidatorExpression {
	private final FormatFactory factory;

	/**
	 * コンストラクタ。
	 * 
	 * @param factory
	 *            フォーマットのファクトリ。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public ParseValidator(final FormatFactory factory) {
		CheckUtils.checkNotNull(factory);
		this.factory = factory;
	}

	/**
	 * コンストラクタ。 format を{@link Format#clone()}で複製して利用する。
	 * 
	 * @param format
	 *            フォーマット。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public ParseValidator(final Format format) {
		this(new CloneFormatFactory(format));
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link ValidatorContext#getTmpValue()}が null の場合。
	 * @throws ClassCastException
	 *             {@link ValidatorContext#getTmpValue()}が文字列でない場合。
	 */
	public boolean interpret(final ValidatorContext context) {
		final String v = (String) context.getTmpValue();
		try {
			context.setTmpValue(factory.newInstance().parseObject(v));
			return true;
		} catch (final ParseException e) {
			return false;
		}
	}

	public FormatFactory getFactory() {
		return factory;
	}
}
