/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.LinkedList;

import woolpack.utils.CheckUtils;

/**
 * {@link #interpret(ValidatorContext)}が呼び出された
 * 回数を数えるテスト用の{@link ValidatorExpression}。
 * このクラスはイミュータブルでもスレッドセーフでもない。
 * {@link #interpret(ValidatorContext)}では
 * {@link ValidatorContext}に対して状態を変更する操作を行わない。
 * 適用しているパターン：Proxy。
 * 
 * @author nakamura
 * 
 */
public class CountValidator implements ValidatorExpression {
	private static final int MAX = 4;

	private final int max;

	private final ValidatorExpression expression;

	private final LinkedList<ValidatorContext> contextList;

	private int count;

	/**
	 * コンストラクタ。
	 * 
	 * @param max
	 *            一覧に保持する{@link ValidatorContext}の最大件数。
	 * @param expression
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public CountValidator(final int max, final ValidatorExpression expression) {
		CheckUtils.checkNotNull(expression);

		this.max = max;
		this.expression = expression;
		this.contextList = new LinkedList<ValidatorContext>();
		count = 0;
	}

	/**
	 * コンストラクタ。 一覧に保持する{@link ValidatorContext}の最大件数を 4 とする。
	 * 
	 * @param expression
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public CountValidator(final ValidatorExpression expression) {
		this(MAX, expression);
	}

	public boolean interpret(final ValidatorContext context) {
		count++;
		if (max > 0) {
			if (count > max) {
				contextList.removeFirst();
			}
			contextList.add(context);
		}
		return expression.interpret(context);
	}

	/**
	 * {@link #interpret(ValidatorContext)}が呼び出された回数を返す。
	 * 
	 * @return {@link #interpret(ValidatorContext)}が呼び出された回数。
	 */
	public int getCount() {
		return count;
	}

	/**
	 * 呼び出された{@link #interpret(ValidatorContext)}
	 * の引数{@link ValidatorContext}の一覧を返す。
	 * 
	 * @return 実行された{@link #interpret(ValidatorContext)}の引数の一覧。
	 */
	public LinkedList<ValidatorContext> getContextList() {
		return contextList;
	}

	public ValidatorExpression getExpression() {
		return expression;
	}
}
