/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * {@link Switchable}の生成器。 コンストラクタツリー内で{@link Switchable}の内容を生成する際に使用する。
 * 本クラスはイミュータブルではないが{@link #get()}の値はイミュータブルである。
 * 
 * @author nakamura
 * 
 * @param <K>
 *            キー。
 * @param <V>
 *            値。
 */
public class SwitchBuilder<K, V> {
	private Map<K, V> map;

	private Map<K, V> checkedMap;

	private V beforeValue;

	/**
	 * コンストラクタ。
	 * 
	 */
	public SwitchBuilder() {
		init();
	}

	private void init() {
		map = null;
		checkedMap = null;
		beforeValue = null;
	}

	private Map<K, V> getMap() {
		if (map == null) {
			map = new LinkedHashMap<K, V>();
		}
		return map;
	}

	private Map<K, V> getCheckedMap() {
		if (checkedMap == null) {
			checkedMap = UtilsConstants.notNullMap(UtilsConstants
					.unoverwritableMap(getMap()));
		}
		return checkedMap;
	}

	/**
	 * キーに対応する値を設定する。
	 * 
	 * @param key
	 *            キー。
	 * @param value
	 *            値。
	 * @return このオブジェクトへの参照。
	 * @throws IllegalStateException
	 *             登録済のキーを指定した場合。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public SwitchBuilder<K, V> put(final K key, final V value) {
		getCheckedMap().put(key, value);
		beforeValue = value;
		return this;
	}

	/**
	 * キーに対応する値を設定する。 前回実行した{@link #put(Object, Object)}の値を値として格納する。
	 * 
	 * @param key
	 *            キー。
	 * @return このオブジェクトへの参照。
	 * @throws IllegalStateException
	 *             登録済のキーを指定した場合。
	 * @throws NullPointerException
	 *             引数が null または初期化後に{@link #put(Object, Object)}を実行していない場合。
	 */
	public SwitchBuilder<K, V> put(final K key) {
		getCheckedMap().put(key, beforeValue);
		return this;
	}

	/**
	 * {@link Switchable}を返し、本インスタンスを初期化する。
	 * 
	 * @return {@link Switchable}。
	 */
	public Switchable<K, V> get() {
		return get(null);
	}

	/**
	 * {@link Switchable}を返し、本インスタンスを初期化する。
	 * 
	 * @param defaultValue
	 *            委譲先の返却値が null の場合のデフォルト値。
	 * @return {@link Switchable}を返し、本インスタンスを初期化する。
	 */
	public Switchable<K, V> get(final V defaultValue) {
		final Map<K, V> localMap = getMap();
		final Collection<K> c = Collections.unmodifiableCollection(localMap
				.keySet());
		init();
		return new Switchable<K, V>() {
			public V get(final K key) {
				final V v = localMap.get(key);
				if (v != null) {
					return v;
				}
				return defaultValue;
			}

			public Collection<K> keys() {
				return c;
			}

			public V getDefault() {
				return defaultValue;
			}
		};
	}
}
