/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

/**
 * 静的メソッドの集まり。
 * 
 * @author nakamura
 * 
 */
public final class PropertyUtils {

	private PropertyUtils() {
	} // カバレージがここを通過してはいけない

	/**
	 * DB における名称を java における名称(先頭小文字)に変換する。
	 * 
	 * @param s
	 *            DB における名称。
	 * @return java における名称(先頭小文字)。
	 */
	public static String toJavaName(final String s) {
		return toJavaName(s, '_');
	}

	/**
	 * DB における名称を java における名称(先頭小文字)に変換する。
	 * 
	 * @param s
	 *            DB における名称。
	 * @param separator
	 *            セパレータ。
	 * @return java における名称(先頭小文字)。
	 */
	public static String toJavaName(final String s, final char separator) {
		if (s == null || s.indexOf(separator) < 0) {
			if (s != null && s.length() >= 1) {
				return Character.toLowerCase(s.charAt(0)) + s.substring(1);
			}
			return s;
		}
		final StringBuilder sb = new StringBuilder(s);
		boolean separateFlag = false;
		int i = 0;
		while (i < sb.length()) {
			char c = sb.charAt(i);
			if (c == separator) {
				separateFlag = true;
				sb.delete(i, i + 1);
			} else {
				if (separateFlag) {
					c = Character.toUpperCase(c);
					separateFlag = false;
				} else {
					c = Character.toLowerCase(c);
				}
				sb.setCharAt(i, c);
				i++;
			}
		}
		return sb.toString();
	}

	/**
	 * java における名称を DB における名称(大文字)に変換する。
	 * 
	 * @param s
	 *            java における名称。
	 * @return DB における名称(大文字)。
	 */
	public static String toSQLName(final String s) {
		return toSQLName(s, '_');
	}

	/**
	 * java における名称を DB における名称(大文字)に変換する。
	 * 
	 * @param s
	 *            java における名称。
	 * @param separator
	 *            セパレータ。
	 * @return DB における名称(大文字)。
	 */
	public static String toSQLName(final String s, final char separator) {
		if (s == null || s.length() == 0) {
			return s;
		}
		final StringBuilder sb = new StringBuilder();
		final int length = s.length();
		boolean upper = false;
		for (int i = length - 1; i >= 0; i--) {
			final char c = s.charAt(i);
			if (Character.isLowerCase(c) && upper) {
				sb.insert(0, separator);
			}
			upper = Character.isUpperCase(c);
			sb.insert(0, Character.toUpperCase(c));
		}
		return sb.toString();
	}
}
