/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.Iterator;
import java.util.Map;

/**
 * {@link #put(String, V)}の際に new String(String) に置き換えて委譲先に設定する{@link Map}。
 * 委譲先に{@link java.util.WeakHashMap}を使用することにより、
 * Stringをキーにした場合にも{@link java.util.WeakHashMap}で
 * キーと値の組を GC させることが可能になる。
 * 適用しているパターン：Decorator。
 * 参考資料：
 * http://www.java-tips.org/java-se-tips/java.util/using-weakhashmap-for-listener-lists.html
 * 
 * @author nakamura
 *
 * @param <V> 値。
 */
public class NewStringKeyMap<V>  extends AbstractKeyIteratorMap<String, V> {
	
	private final Map<String, V> map;

	/**
	 * コンストラクタ。
	 * @param map 委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public NewStringKeyMap(final Map<String, V> map) {
		super();
		CheckUtils.checkNotNull(map);
		this.map = map;
	}

	@Override
	protected Iterator<String> getKeyIterator() {
		return map.keySet().iterator();
	}

	@Override
	protected V getValue(final Object key) {
		return map.get(key);
	}
	
	@Override public V get(final Object key) {
		return getValue(key);
	}
	
	@Override
	public V put(final String key, final V value) {
		return map.put(new String(key), value);
	}

}
