/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.DecimalFormat;
import java.text.FieldPosition;
import java.text.NumberFormat;
import java.text.ParsePosition;

import woolpack.utils.CheckUtils;

/**
 * 数値フォーマット処理を委譲する変換器。
 * 
 * @author nakamura
 * 
 */
public class DelegateNumberFormat extends NumberFormat {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private NumberFormat numberFormat;

	/**
	 * コンストラクタ。
	 * 
	 * @param format
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public DelegateNumberFormat(final NumberFormat format) {
		super();
		CheckUtils.checkNotNull(format);
		this.numberFormat = format;
	}

	/**
	 * コンストラクタ。 {@link DecimalFormat}を委譲先とする。
	 * 
	 */
	public DelegateNumberFormat() {
		this(new DecimalFormat());
	}

	@Override
	public StringBuffer format(final double number,
			final StringBuffer toAppendTo, final FieldPosition pos) {
		return numberFormat.format(number, toAppendTo, pos);
	}

	@Override
	public StringBuffer format(final long number,
			final StringBuffer toAppendTo, final FieldPosition pos) {
		return numberFormat.format(number, toAppendTo, pos);
	}

	@Override
	public Number parse(
			final String source,
			final ParsePosition parsePosition) {
		return numberFormat.parse(source, parsePosition);
	}
	
	@Override
	public Object clone() {
		final DelegateNumberFormat other = (DelegateNumberFormat) super.clone();
		other.numberFormat = (NumberFormat) other.numberFormat.clone();
		return other;
	}

	public NumberFormat getNumberFormat() {
		return numberFormat;
	}
}
