/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.locale;

import java.io.IOException;
import java.io.Reader;
import java.util.Locale;

import woolpack.utils.CheckUtils;
import woolpack.utils.ReaderFactory;
import woolpack.utils.Switchable;

/**
 * スレッドの{@link Locale}により委譲先を分岐する{@link ReaderFactory}。
 * {@link Locale}毎に文字セットを切り替える際に使用することを想定している。
 * 適用しているパターン：フィルタ。
 * 
 * @author nakamura
 * 
 */
public class LocaleReaderFactory implements ReaderFactory {
	private final Switchable<Locale, ReaderFactory> switchable;

	private final ThreadLocal<Locale> threadLocal;

	/**
	 * コンストラクタ。
	 * 
	 * @param switchable
	 *            {@link Locale}と委譲先の{@link Switchable}。
	 * @param threadLocal
	 *            現在のスレッドの{@link Locale}を保持する
	 *            {@link ThreadLocal}。本クラスはこの引数の状態を変化させない。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public LocaleReaderFactory(
			final Switchable<Locale, ReaderFactory> switchable,
			final ThreadLocal<Locale> threadLocal) {
		CheckUtils.checkNotNull(switchable);
		CheckUtils.checkNotNull(threadLocal);
		this.switchable = switchable;
		this.threadLocal = threadLocal;
	}

	/**
	 * @throws NullPointerException
	 *             ロケールに対する{@link ReaderFactory}が存在しない場合。
	 */
	public Reader newInstance(final String id) throws IOException {
		return switchable.get(threadLocal.get()).newInstance(id);
	}
}
