/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;

import org.w3c.dom.Element;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.el.GettingEL;
import woolpack.utils.CheckUtils;
import woolpack.utils.UtilsConstants;

/**
 * 子ノードに HTML の隠し項目(hidden パラメータ)を追加する{@link DomExpression}。
 * 
 * @author nakamura
 * 
 */
public class HiddenAllToChild implements DomExpression {
	private final GettingEL mapEL;

	private final Collection<String> excludeProperties;

	/**
	 * コンストラクタ。
	 * 
	 * @param mapEL
	 *            hidden にする情報が格納された{@link Map}への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param excludeProperties
	 *            hidden にしないキーの一覧。本クラスはこの引数の状態を変化させない。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public HiddenAllToChild(final GettingEL mapEL,
			final Collection<String> excludeProperties) {
		CheckUtils.checkNotNull(mapEL);
		CheckUtils.checkNotNull(excludeProperties);

		this.mapEL = mapEL;
		this.excludeProperties = excludeProperties;
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link DomContext#getNode()}または取得した{@link Map}が null
	 *             の場合。
	 */
	public void interpret(final DomContext context) {
		final Map map = (Map) mapEL.getValue(context);
		for (final Object entryObject : map.entrySet()) {
			final Entry entry = (Entry) entryObject;
			if (excludeProperties.contains(entry.getKey())) {
				continue;
			}
			final Iterable values = UtilsConstants.toIterable(entry.getValue());
			for (final Object value : values) {
				final Element e = DomConstants.getDocumentNode(
						context.getNode()).createElement("INPUT");
				e.setAttribute("type", "hidden");
				e.setAttribute("name", entry.getKey().toString());
				e.setAttribute("value", value.toString());
				context.getNode().appendChild(e);
			}
		}
	}
}
