/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.utils.CheckUtils;

/**
 * 各属性値の出現回数(1回か2回以上)により処理を分岐する{@link DomExpression}。 適用しているパターン：フィルタ。
 * 
 * @author nakamura
 * 
 */
public class BranchPropertyCount extends AbstractCountProperty {
	private final DomExpression firstExpression;

	private final DomExpression pluralExpression;

	/**
	 * コンストラクタ。
	 * 
	 * @param key
	 *            プロパティの出現回数を保持する{@link DomContext#getLocal()}上の位置。
	 * @param attrNames
	 *            属性名の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @param firstExpression
	 *            最初の検索結果に対する委譲先。
	 * @param pluralExpression
	 *            2番目以降の検索結果に対する委譲先。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             key が空の場合。
	 */
	public BranchPropertyCount(final String key,
			final Iterable<String> attrNames,
			final DomExpression firstExpression,
			final DomExpression pluralExpression) {
		super(key, attrNames);
		CheckUtils.checkNotNull(firstExpression);
		CheckUtils.checkNotNull(pluralExpression);
		this.firstExpression = firstExpression;
		this.pluralExpression = pluralExpression;
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param attrNames
	 *            属性名の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @param firstExpression
	 *            最初の検索結果に対する委譲先。
	 * @param pluralExpression
	 *            2番目以降の検索結果に対する委譲先。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public BranchPropertyCount(final Iterable<String> attrNames,
			final DomExpression firstExpression,
			final DomExpression pluralExpression) {
		this(KEY, attrNames, firstExpression, pluralExpression);
	}

	@Override
	public void interpret(final DomContext context,
			final PropertyCounter counter) {
		if (counter.getCount() == 1) {
			firstExpression.interpret(context);
		} else {
			pluralExpression.interpret(context);
		}
	}
}
