/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.utils.CheckUtils;

/**
 * 各属性値の出現回数をカウントして
 * {@link #interpret(DomContext, PropertyCounter)}
 * に処理を委譲する{@link DomExpression}。
 * 適用しているパターン：Template Method。
 * 
 * @author nakamura
 * 
 */
public abstract class AbstractCountProperty implements DomExpression {
	/**
	 * プロパティの出現回数を保持する{@link DomContext#getLocal()}上の位置のデフォルト値。
	 */
	public static final String KEY
	= "woolpack.html.IF_DUPLICATE_PROPERTY_COUNTER";

	private final String key;

	private final Iterable<String> attrNames;

	/**
	 * コンストラクタ。
	 * 
	 * @param key
	 *            プロパティの出現回数を保持する{@link DomContext#getLocal()}上の位置。
	 * @param attrNames
	 *            属性名の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             key が空の場合。
	 */
	public AbstractCountProperty(
			final String key,
			final Iterable<String> attrNames) {
		CheckUtils.checkNotEmpty(key);
		CheckUtils.checkNotNull(attrNames);
		this.key = key;
		this.attrNames = attrNames;
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param attrNames
	 *            属性名の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public AbstractCountProperty(final Iterable<String> attrNames) {
		this(KEY, attrNames);
	}

	public void interpret(final DomContext context) {
		if (context.getNode().getNodeType() != Node.ELEMENT_NODE) {
			return;
		}
		PropertyCounter counter = (PropertyCounter) context.getLocal().get(key);
		if (counter == null) {
			counter = new PropertyCounter();
			context.getLocal().put(key, counter);
		}
		final Element element = (Element) context.getNode();
		for (final String attrName : attrNames) {
			if (element.hasAttribute(attrName)) {
				counter.put(attrName, element.getAttribute(attrName));
				interpret(context, counter);
			}
		}
	}

	/**
	 * {@link #interpret(DomContext)}から呼び出される(called)。
	 * 実装クラスでは{@link PropertyCounter}の状態により処理を行う必要がある。
	 * 
	 * @param context
	 *            コンテキスト。
	 * @param counter
	 *            プロパティの出現回数をカウントするカウンタ。
	 */
	public abstract void interpret(DomContext context, PropertyCounter counter);
}
