/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import woolpack.utils.CheckUtils;

/**
 * {@link PropertyELFactory}の取得結果に対し作用させる{@link EL}。
 * 
 * @author nakamura
 * 
 */
public class PropertyEL extends AbstractEL {
	private final String propertyName;

	private final PropertyELFactory factory;

	/**
	 * コンストラクタ。
	 * 
	 * @param propertyName
	 *            プロパティ名。
	 * @param factory
	 *            プロパティにアクセスする{@link GettingEL}と{@link EL}のファクトリ。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             引数のいずれかが空の場合。
	 */
	public PropertyEL(
			final String propertyName,
			final PropertyELFactory factory) {
		super();
		CheckUtils.checkNotEmpty(propertyName);
		CheckUtils.checkNotNull(factory);

		this.propertyName = propertyName;
		this.factory = factory;
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param propertyName
	 *            プロパティ名。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             引数が空の場合。
	 */
	public PropertyEL(final String propertyName) {
		this(propertyName, ELConstants.PROPERTY_EL_FACTORY);
	}

	@Override
	public Object getValue(final Object root, final Class clazz) {
		return factory.newGetter(root.getClass(), propertyName).getValue(root,
				clazz);
	}

	@Override
	public void setValue(final Object root, final Object value) {
		factory.newSetter(root.getClass(), propertyName).setValue(root, value);
	}

	public PropertyELFactory getFactory() {
		return factory;
	}

	public String getPropertyName() {
		return propertyName;
	}
}
