/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.util.Map;

import woolpack.el.convert.Convertable;
import woolpack.utils.CheckUtils;

/**
 * {@link Map}のみをサポートする{@link PropertyELFactory}。 適用しているパターン：Balking。
 * 
 * @author nakamura
 * 
 */
public class MapPropertyELFactory implements PropertyELFactory {

	private final Convertable convertable;

	private final PropertyELFactory factory;

	/**
	 * コンストラクタ。
	 * 
	 * @param convertable
	 *            変換器。
	 * @param factory
	 *            型が{@link Map}でない場合の委譲先。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public MapPropertyELFactory(final Convertable convertable,
			final PropertyELFactory factory) {
		CheckUtils.checkNotNull(convertable);
		CheckUtils.checkNotNull(factory);

		this.convertable = convertable;
		this.factory = factory;
	}

	public GettingEL newGetter(final Class clazz, final String propertyName) {
		if (Map.class.isAssignableFrom(clazz)) {
			return new MapEL(propertyName, convertable);
		}
		return factory.newGetter(clazz, propertyName);
	}

	public EL newSetter(final Class clazz, final String propertyName) {
		if (Map.class.isAssignableFrom(clazz)) {
			return new MapEL(propertyName, convertable);
		}
		return factory.newSetter(clazz, propertyName);
	}

	public Convertable getConvertable() {
		return convertable;
	}

	public PropertyELFactory getFactory() {
		return factory;
	}
}
