/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import woolpack.el.ArrayPathEL;
import woolpack.el.GettingEL;
import woolpack.el.MapEL;
import woolpack.utils.CheckUtils;

/**
 * ネットワークアドレスを返す{@link Evaluable}。ネットワークアドレス毎にセキュリティ設定を定義するために使用することを想定している。
 * 
 * @author nakamura
 * 
 */
public class EvalNetMask implements Evaluable<String> {
	private static final int IP_COUNT = 4;

	private static final Pattern PATTERN = Pattern
			.compile("(\\d+)\\.(\\d+)\\.(\\d+)\\.(\\d+)");

	/**
	 * アドレスの取得先のデフォルト値。
	 */
	public static final GettingEL KEY = new ArrayPathEL(DomConstants.LOCAL_EL,
			new MapEL("woolpack.dom.NET_MASK"));

	private final GettingEL addressEL;

	private final int[] maskArray;

	/**
	 * コンストラクタ。
	 * 
	 * @param addressEL
	 *            アドレスの取得先。
	 * @param netMask
	 *            ネットマスク。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws IllegalArgumentException
	 *             引数が不正の場合。
	 */
	public EvalNetMask(final GettingEL addressEL, final String netMask) {
		CheckUtils.checkNotNull(addressEL);
		CheckUtils.checkNotNull(netMask);

		this.maskArray = toArray(netMask);
		if (maskArray == null) {
			throw new IllegalArgumentException();
		}
		this.addressEL = addressEL;
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param netMask
	 *            ネットマスク。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 * @throws IllegalArgumentException
	 *             引数が不正の場合。
	 */
	public EvalNetMask(final String netMask) {
		this(KEY, netMask);
	}

	private static int[] toArray(final String s) {
		if (s == null) {
			return null;
		}
		final Matcher m = PATTERN.matcher(s);
		if (!m.matches()) {
			return null;
		}
		final int[] a = new int[IP_COUNT];
		for (int i = 0; i < a.length; i++) {
			a[i] = Integer.parseInt(m.group(i + 1));
		}
		return a;
	}

	public String eval(final DomContext context) {
		final String address = (String) addressEL.getValue(context,
				String.class);
		final int[] a = toArray(address);
		if (a == null) {
			return null;
		}

		for (int i = 0; i < a.length; i++) {
			a[i] = a[i] & maskArray[i];
		}

		final StringBuilder sb = new StringBuilder();
		for (int i = 0; i < a.length; i++) {
			if (i != 0) {
				sb.append('.');
			}
			sb.append(a[i]);
		}

		return sb.toString();
	}

}
