/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import org.w3c.dom.Element;

import woolpack.utils.CheckUtils;
import woolpack.utils.Switchable;
import woolpack.utils.UtilsConstants;

/**
 * DOM エレメントの属性値で委譲先を分岐する{@link DomExpression}。 適用しているパターン：フィルタ。
 * 
 * @author nakamura
 * 
 */
public class BranchByAttrValue implements DomExpression {
	private final Iterable<String> attrNames;

	private final Switchable<String, DomExpression> switchable;

	/**
	 * コンストラクタ。
	 * 
	 * @param attrNames
	 *            属性名の一覧。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @param switchable
	 *            属性値と委譲先の{@link Switchable}。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public BranchByAttrValue(final Iterable<String> attrNames,
			final Switchable<String, DomExpression> switchable) {
		CheckUtils.checkNotNull(attrNames);
		CheckUtils.checkNotNull(switchable);

		this.attrNames = attrNames;
		this.switchable = UtilsConstants.nullToDefaultSwitchIfAbsent(
				DomConstants.NULL, switchable);
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link DomContext#getNode()}または
	 *             {@link DomContext#getId()}に対応する値が null の場合。
	 * @throws ClassCastException
	 *             {@link DomContext#getNode()}が{@link Element}でない場合。
	 * 
	 */
	public void interpret(final DomContext context) {
		final Element element = (Element) context.getNode();
		for (final String attrName : attrNames) {
			final String value = element.getAttribute(attrName);
			if (switchable.keys().contains(value)) {
				switchable.get(value).interpret(context);
				return;
			}
		}
		switchable.getDefault().interpret(context);
	}
}
