/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.container;

import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import woolpack.utils.AbstractKeyIteratorMap;
import woolpack.utils.CheckUtils;

/**
 * オブジェクトコンテナを表す{@link Map}。 スコープを制御する機能のみを具備する。 適用しているパターン：Adapter。
 * 
 * @author nakamura
 * 
 */
public class ScopeContainer extends AbstractKeyIteratorMap<String, Object> {
	/**
	 * スコープ管理されたオブジェクトを格納する request と session と application 上の位置の接頭辞のデフォルト値。
	 */
	public static final String KEY_PREFIX
	= "woolpack.container.SCOPE_CONTAINER.";

	private final String keyPrefix;

	private final Map<String, Object> request;

	private final ConcurrentMap<String, Object> session;

	private final ConcurrentMap<String, Object> application;

	private final Map<String, AbstractComponentDef> componentDefs;

	/**
	 * コンストラクタ。
	 * 
	 * @param keyPrefix
	 *            スコープ管理されたオブジェクトを格納する request と session と application 上の位置の接頭辞。
	 * @param request
	 *            Request スコープ。本クラスはこの引数の状態を変化させる。
	 * @param session
	 *            Session スコープ。本クラスはこの引数の状態を変化させる。
	 * @param application
	 *            Application スコープ。本クラスはこの引数の状態を変化させる。
	 * @param componentDefs
	 *            コンポーネント定義の{@link Map}。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 * @throws StringIndexOutOfBoundsException
	 *             attrNameが空の場合。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public ScopeContainer(final String keyPrefix,
			final Map<String, Object> request,
			final ConcurrentMap<String, Object> session,
			final ConcurrentMap<String, Object> application,
			final Map<String, AbstractComponentDef> componentDefs) {
		super();
		CheckUtils.checkNotEmpty(keyPrefix);
		CheckUtils.checkNotNull(componentDefs);
		CheckUtils.checkNotNull(request);
		CheckUtils.checkNotNull(session);
		CheckUtils.checkNotNull(application);

		this.keyPrefix = keyPrefix;
		this.request = request;
		this.session = session;
		this.application = application;
		this.componentDefs = componentDefs;
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param request
	 *            Request スコープ。本クラスはこの引数の状態を変化させる。
	 * @param session
	 *            Session スコープ。本クラスはこの引数の状態を変化させる。
	 * @param application
	 *            Application スコープ。本クラスはこの引数の状態を変化させる。
	 * @param componentDefs
	 *            コンポーネント定義の{@link Map}。本クラスはこの引数の状態を変化させない。ステートレスであるべき。
	 */
	public ScopeContainer(final Map<String, Object> request,
			final ConcurrentMap<String, Object> session,
			final ConcurrentMap<String, Object> application,
			final Map<String, AbstractComponentDef> componentDefs) {
		this(KEY_PREFIX, request, session, application, componentDefs);
	}

	@Override
	protected Iterator<String> getKeyIterator() {
		return Collections.unmodifiableSet(componentDefs.keySet()).iterator();
	}

	@Override
	protected Object getValue(final Object key) {
		final String keyString = (String) key;
		final AbstractComponentDef componentDef = componentDefs.get(keyString);
		final Object result = componentDef.getComponent(keyPrefix + keyString,
				this);
		// TODO この位置で result に inject
		// する。(request/session/application(thisでもよい)を渡す)
		return result;
	}

	/**
	 * @throws NullPointerException
	 *             keyに対応する{@link AbstractComponentDef}が定義されていない場合。
	 * @throws ClassCastException
	 *             keyが文字列型でない場合。
	 * @throws UnsupportedOperationException
	 *             いずれかのスコープへの設定に失敗した場合。
	 * @throws RuntimeException
	 *             {@link AbstractComponentDef#newInstance()}が例外を投げた場合。
	 */
	@Override
	public Object get(final Object key) {
		return getValue(key);
	}

	ConcurrentMap<String, Object> getApplication() {
		return application;
	}

	Map<String, Object> getRequest() {
		return request;
	}

	ConcurrentMap<String, Object> getSession() {
		return session;
	}
}
