/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.config;

import java.util.Enumeration;
import java.util.ResourceBundle;

import woolpack.utils.CheckUtils;

/**
 * {@link ResourceBundle}を読み込み
 * {@link ConfigContext#getMap()}に追加する{@link ConfigExpression}。
 * 適用しているパターン：Adapter, Template Method。
 * 
 * @author nakamura
 * 
 */
public class PutResourceBundle implements ConfigExpression {
	private final String path;

	/**
	 * コンストラクタ。
	 * 
	 * @param path
	 *            {@link ResourceBundle}の識別子。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             引数が空の場合。
	 */
	public PutResourceBundle(final String path) {
		CheckUtils.checkNotEmpty(path);
		this.path = path;
	}

	/**
	 * @throws NullPointerExcpeption
	 *             引数または{@link ConfigContext#getMap()}が null の場合。
	 * @throws java.util.MissingResourceException
	 *             リソースバンドルが見つからない場合。
	 * @throws UnsupportedOperationException
	 *             メッセージキーの設定に失敗した場合。
	 */
	public void interpret(final ConfigContext context) {
		final ResourceBundle b = getResourceBundle(path);
		for (final Enumeration<String> e = b.getKeys(); e.hasMoreElements();) {
			final String key = e.nextElement();
			context.getMap().put(key, b.getObject(key));
		}
	}

	/**
	 * {@link ResourceBundle}を取得するために
	 * {@link #interpret(ConfigContext)}から呼び出される(called)。
	 * デフォルトは{@link ResourceBundle#getBundle(java.lang.String)}の結果を返す。
	 * {@link java.util.Locale}や端末ごとに{@link ResourceBundle}を生成する際にオーバライドする。
	 * オーバライドする際は {@link CacheMap#getCacheKey()} と整合性を保つ必要がある。
	 * 
	 * @param basePath
	 *            {@link ResourceBundle}の識別子。
	 * @return {@link ResourceBundle}のインスタンス。
	 */
	public ResourceBundle getResourceBundle(final String basePath) {
		return ResourceBundle.getBundle(basePath);
	}
}
