/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import junit.framework.TestCase;

public class KeyIteratorMapTest extends TestCase {
	private final Map<String,String> actualMap = new HashMap<String,String>();
	private final Map<String,String> map = new KeyIteratorMap<String,String>(){
		protected Iterator<String> getKeyIterator() {
			return actualMap.keySet().iterator();
		}
		protected String getValue(final Object key) {
			return actualMap.get(key);
		}
	};
	
	@Override protected void setUp() throws Exception {
		super.setUp();
		actualMap.put("1", "value1");
		actualMap.put("2", "value2");
	}

	public void testClear(){
		map.clear();
		assertTrue(map.isEmpty());
	}

	public void testSize(){
		assertEquals(2, map.size());
	}
	
	public void testIsEmpty(){
		assertFalse(map.isEmpty());
	}
	
	public void testContainsKey(){
		assertTrue(map.containsKey("1"));
		assertFalse(map.containsKey("0"));
	}
	
	public void testContainsValue(){
		assertTrue(map.containsValue("value1"));
		assertFalse(map.containsValue("value0"));
	}
	
	public void testGet(){
		assertEquals("value1", map.get("1"));
		assertNull(map.get("0"));
	}
	
	public void testPut(){
		try{
			map.put("key", "value");
			fail();
		}catch(UnsupportedOperationException expected){
		}
	}
	
	public void testRemove(){
		map.remove("1");
		assertEquals(1, map.size());
		assertEquals("value2", map.get("2"));
	}
	
	public void testPutAll(){
		try{
			final Map<String,String> addMap = new HashMap<String,String>();
			addMap.put("key", "value");
			map.putAll(addMap);
			fail();
		}catch(UnsupportedOperationException expected){
		}
	}
	
	public void testKeySet(){
		final Set<String> set = map.keySet();
		final Set<String> expected = new HashSet<String>();
		expected.add("1");
		expected.add("2");
		assertEquals(expected, set);
		
		for(final Iterator<String> it = set.iterator();it.hasNext();){
			final String key = it.next();
			if(key.equals("1")){
				assertEquals("value1", map.get("1"));
			}else if(key.equals("2")){
				assertEquals("value2", map.get("2"));
			}else{
				fail();
			}
		}
	}

	public void testValues(){
		final Collection<String> c = map.values();
		final Set<String> expected = new HashSet<String>();
		expected.add("value1");
		expected.add("value2");
		assertTrue(c.containsAll(expected));
		assertTrue(expected.containsAll(c));
	}
	
	public void testEntrySet(){
		final Set<Map.Entry<String,String>> set = map.entrySet();
		assertEquals(2, set.size());
		
		final Map<String,String> tmpMap = new HashMap<String,String>();
		for(final Map.Entry<String,String> e:set){
			tmpMap.put(e.getKey(), e.getValue());
			try{
				e.setValue("VALUE");
				fail();
			}catch(UnsupportedOperationException expected){
			}
		}
		assertEquals(actualMap, tmpMap);
	}
}
