/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.Format;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;

public class LimitedValueFormatTest extends TestCase {

	public void testConstructor(){
		try{
			new LimitedValueFormat(null, "defaultParsedValue", "defaultFormattedValue");
			fail();
		}catch(final NullPointerException expected){
		}
		new LimitedValueFormat(new HashMap<String,Object>(), null, "defaultParsedValue");
		new LimitedValueFormat(new HashMap<String,Object>(), "defaultFormattedValue", "defaultParsedValue");
		new LimitedValueFormat(new HashMap<String,Object>(), "defaultFormattedValue", null);
	}
	
	public void testDuplicate0(){
		final Map<String,String> m = new HashMap<String,String>();
		m.put("key0", "value0");
		m.put("key1", "value1");
		m.put("key2", "value1");
		try{
			new LimitedValueFormat(m, "defaultFormattedValue", "defaultParsedValue");
			fail();
		}catch(final IllegalStateException expected){
			assertEquals("already registered: value1", expected.getMessage());
		}
	}
	
	private void scenarioParse(final Format format) throws ParseException{
		assertEquals("value0", format.parseObject("key0"));
		assertEquals("value1", format.parseObject("key1"));
		assertEquals("value2", format.parseObject("key2"));
	}

	private void scenarioParseException(final Format format) throws ParseException{
		try{
			format.parseObject("key3");
			fail();
		}catch(final ParseException expected){
		}
	}

	private void scenarioParseDefault(final Format format) throws ParseException{
		assertEquals("defaultParsedValue", format.parseObject("key3"));
	}

	private void scenarioFormat(final Format format) throws ParseException{
		assertEquals("key0", format.format("value0"));
		assertEquals("key1", format.format("value1"));
		assertEquals("key2", format.format("value2"));
	}

	private void scenarioFormatNothing(final Format format) throws ParseException{
		assertEquals("key3", format.format("key3"));
	}

	private void scenarioFormatDefault(final Format format) throws ParseException{
		assertEquals("defaultFormattedValue", format.format("value3"));
	}

	public void testNormal() throws ParseException{
		final Map<String,String> m0 = new HashMap<String,String>();
		m0.put("key0", "value0");
		m0.put("key1", "value1");
		m0.put("key2", "value2");
		{
			final Format format0 = new LimitedValueFormat(m0, null, null);
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatNothing(format0);
			
			final Format format1 = (Format)format0.clone();
			assertTrue(format0 != format1);
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatNothing(format0);
		}
		{
			final Format format0 = new LimitedValueFormat(m0, "defaultParsedValue", "defaultFormattedValue");
			scenarioParse(format0);
			scenarioParseDefault(format0);
			scenarioFormat(format0);
			scenarioFormatDefault(format0);
			
			final Format format1 = (Format)format0.clone();
			assertTrue(format0 != format1);
			scenarioParse(format0);
			scenarioParseDefault(format0);
			scenarioFormat(format0);
			scenarioFormatDefault(format0);
		}
	}
}
