/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.locale;

import java.util.HashMap;
import java.util.Locale;

import junit.framework.TestCase;

import org.w3c.dom.Node;

import woolpack.dom.Branch;
import woolpack.dom.Count;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.test.TestUtils;
import woolpack.utils.SwitchBuilder;

public class LocaleCacheNodeTest extends TestCase {

	public void testConstructor(){
		try{
			new LocaleCacheNode(new HashMap<Object,Node>(), null, new ThreadLocal<Locale>());
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new LocaleCacheNode(new HashMap<Object,Node>(), DomConstants.NULL, null);
			fail();
		}catch(final NullPointerException expected){
		}
		new LocaleCacheNode(null, DomConstants.NULL, new ThreadLocal<Locale>());
	}

	
	public void testCacheOn(){
		final ThreadLocal<Locale> threadLocal = new ThreadLocal<Locale>();
		final Count counter0 = new Count(
				new BranchByLocale(
						new SwitchBuilder<Locale,DomExpression>()
						.put(Locale.JAPANESE, new Branch<String>(DomConstants.EVAL_ID, 
								new SwitchBuilder<String,DomExpression>()
								.put("id0", TestUtils.getToNodeBody("id0j"))
								.put("id1", TestUtils.getToNodeBody("id1j"))
								.get()))
						.put(Locale.ENGLISH, new Branch<String>(DomConstants.EVAL_ID, 
								new SwitchBuilder<String,DomExpression>()
								.put("id0", TestUtils.getToNodeBody("id0e"))
								.put("id1", TestUtils.getToNodeBody("id1e"))
								.get()))
						.get(new Branch<String>(DomConstants.EVAL_ID, 
								new SwitchBuilder<String,DomExpression>()
								.put("id0", TestUtils.getToNodeBody("id0n"))
								.put("id1", TestUtils.getToNodeBody("id1n"))
								.get())
						), threadLocal)
		);
		final DomExpression expression = new LocaleCacheNode(new HashMap<Object,Node>(), counter0, threadLocal);
		
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.JAPANESE);
			context.setId("id0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id0j"));
			assertEquals(1, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.JAPANESE);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1j"));
			assertEquals(2, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}		
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.ENGLISH);
			context.setId("id0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id0e"));
			assertEquals(3, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.ENGLISH);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1e"));
			assertEquals(4, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.CHINESE);
			context.setId("id0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id0n"));
			assertEquals(5, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(null);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1n"));
			assertEquals(6, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.ENGLISH);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1e"));
			assertEquals(6, counter0.getCount());
			assertTrue(context != counter0.getContextList().getLast());
		}
	}
	
	public void testCacheOff(){
		final ThreadLocal<Locale> threadLocal = new ThreadLocal<Locale>();
		final Count counter0 = new Count(
				new BranchByLocale(
						new SwitchBuilder<Locale,DomExpression>()
						.put(Locale.JAPANESE, new Branch<String>(DomConstants.EVAL_ID, 
								new SwitchBuilder<String,DomExpression>()
								.put("id0", TestUtils.getToNodeBody("id0j"))
								.put("id1", TestUtils.getToNodeBody("id1j"))
								.get()))
						.put(Locale.ENGLISH, new Branch<String>(DomConstants.EVAL_ID, 
								new SwitchBuilder<String,DomExpression>()
								.put("id0", TestUtils.getToNodeBody("id0e"))
								.put("id1", TestUtils.getToNodeBody("id1e"))
								.get()))
						.get(new Branch<String>(DomConstants.EVAL_ID, 
								new SwitchBuilder<String,DomExpression>()
								.put("id0", TestUtils.getToNodeBody("id0n"))
								.put("id1", TestUtils.getToNodeBody("id1n"))
								.get())
						), threadLocal)
		);
		final DomExpression expression = new LocaleCacheNode(null, counter0, threadLocal);
		
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.JAPANESE);
			context.setId("id0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id0j"));
			assertEquals(1, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.JAPANESE);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1j"));
			assertEquals(2, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}		
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.ENGLISH);
			context.setId("id0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id0e"));
			assertEquals(3, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.ENGLISH);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1e"));
			assertEquals(4, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.CHINESE);
			context.setId("id0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id0n"));
			assertEquals(5, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(null);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1n"));
			assertEquals(6, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
		{
			final DomContext context = new DomContext();
			threadLocal.set(Locale.ENGLISH);
			context.setId("id1");
			expression.interpret(context);

			assertTrue(TestUtils.equalsBody(context, "id1e"));
			assertEquals(7, counter0.getCount());
			assertTrue(context == counter0.getContextList().getLast());
		}
	}
}
