/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import junit.framework.TestCase;
import woolpack.dom.Count;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.NodeLoopDelegate;
import woolpack.dom.Serial;
import woolpack.test.TestUtils;

public class BranchPropertyCountTest extends TestCase {

	public void testConstructor(){
		try{
			new BranchPropertyCount(null, Arrays.asList("name"), DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new BranchPropertyCount("", Arrays.asList("name"), DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final StringIndexOutOfBoundsException expected){
		}
		try{
			new BranchPropertyCount("key", null, DomConstants.NULL, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testNormal(){
		final List<String> list0 = new ArrayList<String>();
		final List<String> list1 = new ArrayList<String>();
		final Count counter0 = new Count(new DomExpression(){
			public void interpret(final DomContext context) {
				list0.add(context.getNode().getFirstChild().getNodeValue());
			}
		});
		final Count counter1 = new Count(new DomExpression(){
			public void interpret(final DomContext context) {
				list1.add(context.getNode().getFirstChild().getNodeValue());
			}
		});
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody(
						"<SPAN id=\"name0\">value00</SPAN>" +
						"<SPAN id=\"name1\">value10</SPAN>" +
						"<SPAN id=\"name0\">value01</SPAN>"),
			new NodeLoopDelegate(new BranchPropertyCount(Arrays.asList("id"), counter0, counter1))
		);
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertTrue(TestUtils.equals(Arrays.asList("value00", "value10"), list0));
		assertTrue(TestUtils.equals(Arrays.asList("value01"), list1));
	}
	
	public void testTipsRemoveDuplicateAndInit(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody(
						"<SPAN id=\"name0\">value00</SPAN>" +
						"<SPAN id=\"name1\">value10</SPAN>" +
						"<SPAN id=\"name0\">value01</SPAN>" +
						"<SPAN id=\"name1\">value11</SPAN>"),
			new NodeLoopDelegate(new BranchPropertyCount(
					Arrays.asList("id"), 
					DomConstants.REMOVE_CHILDREN, 
					DomConstants.REMOVE_THIS))
		);
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, 
				"<SPAN id=\"name0\"></SPAN>" +
				"<SPAN id=\"name1\"></SPAN>"));
	}
}
