/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el.convert;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;

import junit.framework.TestCase;
import woolpack.text.BooleanFormat;

public class SimpleConverterTest extends TestCase {

	public void testConstructor(){
		try{
			new SimpleConverter(null, new SimpleDateFormat(), new BooleanFormat());
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new SimpleConverter(new DecimalFormat(), null, new BooleanFormat());
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new SimpleConverter(new DecimalFormat(), new SimpleDateFormat(), null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testNormal() throws ParseException{
		final Convertable c = new SimpleConverter(
				new DecimalFormat(), new SimpleDateFormat("yyyyMMdd"), new BooleanFormat()
		);

		assertEquals("true", c.convert(Boolean.TRUE, String.class, null));
		assertEquals("20060708", c.convert(new SimpleDateFormat("yyyyMMdd").parse("20060708"), String.class, null));
		assertEquals("1", c.convert((byte)1, String.class, null));
		assertEquals("1,000", c.convert((short)1000, String.class, null));
		assertEquals("1,000", c.convert((int)1000, String.class, null));
		assertEquals("1,000", c.convert((long)1000, String.class, null));
		assertEquals("1,000", c.convert((float)1000, String.class, null));
		assertEquals("1,000", c.convert((double)1000, String.class, null));
		assertEquals("1,000", c.convert(BigInteger.valueOf(1000), String.class, null));
		assertEquals("1,000", c.convert(new BigDecimal(1000), String.class, null));
		assertEquals("c", c.convert('c', String.class, null));
		
		assertEquals(Boolean.TRUE, c.convert("true", Boolean.class, null));
		assertEquals(Boolean.FALSE, c.convert("false", Boolean.class, null));
		assertEquals(new Character('a'), c.convert("abc", Character.class, null));
		assertEquals(new java.sql.Date(new SimpleDateFormat("yyyyMMdd").parse("20060708").getTime()), c.convert("20060708", java.sql.Date.class, null));
		assertEquals(new SimpleDateFormat("yyyyMMdd").parse("20060708"), c.convert("20060708", java.util.Date.class, null));
		
		assertEquals(new Byte((byte)1), c.convert("1", Byte.class, null));
		assertEquals(new Short((short)1), c.convert("1", Short.class, null));
		assertEquals(new Integer(1), c.convert("1", Integer.class, null));
		assertEquals(new Long(1), c.convert("1", Long.class, null));
		assertEquals(new Float(1), c.convert("1", Float.class, null));
		assertEquals(new Double(1), c.convert("1", Double.class, null));
		assertEquals(BigInteger.valueOf(1), c.convert("1", BigInteger.class, null));
		assertEquals(new BigDecimal(1), c.convert("1", BigDecimal.class, null));

		assertEquals(new Byte((byte)1), c.convert(new Integer(1), Byte.class, null));
		assertEquals(new Short((short)1), c.convert(new Integer(1), Short.class, null));
		assertEquals(new Integer(1), c.convert(new Long(1), Integer.class, null));
		assertEquals(new Long(1), c.convert(new Integer(1), Long.class, null));
		assertEquals(new Float(1), c.convert(new Integer(1), Float.class, null));
		assertEquals(new Double(1), c.convert(new Integer(1), Double.class, null));
		assertEquals(BigInteger.valueOf(1), c.convert(new Integer(1), BigInteger.class, null));
		assertEquals(new BigDecimal(1), c.convert(new Integer(1), BigDecimal.class, null));
		
		{
			final Object o = new Object();
			assertSame(o, c.convert(o, Object.class, null));
		}
		
		try{
			c.convert("abc", Integer.class, null);
			fail();
		}catch(final RuntimeException e){
			assertTrue(e.getCause() instanceof ParseException);
		}
	}
}
