/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.util.HashMap;
import java.util.Map;

import woolpack.utils.UtilsConstants;

/**
 * 有限個の値をフォーマットする変換器。
 * @author nakamura
 *
 */
public class LimitedValueFormat extends Format {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private final Map<String,? extends Object> parseMap;
	private final Object defaultParsedValue;
	private final String defaultFormattedValue;

	private final Map<? extends Object,String> formatMap;
	
	/**
	 * コピーコンストラクタ。
	 * {@link #clone()}から呼び出される(called)。
	 * @param format コピー元。
	 */
	protected LimitedValueFormat(final LimitedValueFormat format){
		this.parseMap = format.parseMap;
		this.defaultParsedValue = format.defaultParsedValue;
		this.defaultFormattedValue = format.defaultFormattedValue;
		this.formatMap = format.formatMap;
	}
	
	/**
	 * コンストラクタ。
	 * @param parseMap {@link #parseObject(String, ParsePosition)}で使用する{@link Map}。本クラスはこの引数の状態を変化させない。
	 * @param defaultParsedValue parseMap に変更元が定義されていない場合の変更先。null を指定した場合は変更しない。
	 * @param defaultFormattedValue parseMap に変更元が定義されていない場合の変更先。null を指定した場合はを変更しない。
	 * @throws NullPointerException parseMap が null の場合。
	 * @throws IllegalArgumentException parseMap の値が重複している場合。
	 */
	public LimitedValueFormat(
			final Map<String,? extends Object> parseMap,  
			final Object defaultParsedValue,
			final String defaultFormattedValue){
		this.parseMap = new HashMap<String,Object>(parseMap);
		this.defaultParsedValue = defaultParsedValue;
		this.defaultFormattedValue = defaultFormattedValue;
		this.formatMap = UtilsConstants.inverseMap(parseMap);
	}
	
	@Override
	public StringBuffer format(final Object obj, final StringBuffer toAppendTo,
			final FieldPosition pos) {
		final int start = toAppendTo.length();
		String o = formatMap.get(obj);
		if(o == null){
			o = defaultFormattedValue;
		}
		if(o != null){
			toAppendTo.append(o);
		}else{
			toAppendTo.append(obj);
		}
		pos.setBeginIndex(start);
		pos.setEndIndex(toAppendTo.length());
		return toAppendTo;
	}

	@Override
	public Object parseObject(final String source, final ParsePosition pos) {
		for(final String key:parseMap.keySet()){
			if(source.startsWith(key, pos.getIndex())){
				pos.setIndex(pos.getIndex()+key.length());
				return parseMap.get(key);
			}
		}
		if(defaultParsedValue != null){
			pos.setIndex(source.length());
			return defaultParsedValue;
		}
		pos.setErrorIndex(pos.getIndex());
		return null;
	}

	@Override public Object clone(){
		return new LimitedValueFormat(this);
	}

	public String getDefaultFormattedValue() {
		return defaultFormattedValue;
	}

	public Object getDefaultParsedValue() {
		return defaultParsedValue;
	}

	public Map<String, ? extends Object> getParseMap() {
		return parseMap;
	}
}
