/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.text.RegExpFormat;
import woolpack.utils.MapBuilder;

public class ValueLoopValidatorTest extends TestCase {

	public void testConstructor(){
		try{
			new ValueLoopValidator(null, ValidatorConstants.FALSE);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new ValueLoopValidator(ValidatorConstants.ANDAND, null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testInterpretNull(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(new HashMap<String,List<Object>>());
		context.setTmpKey("key0");
		
		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getIntepretCount());
		assertEquals("key0", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
	}

	public void testInterpretObject(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(new HashMap<String,List<Object>>());
		context.setTmpKey("key0");
		
		context.setTmpValue(new Object());
		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getIntepretCount());
		assertEquals("key0", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
	}

	public void testInterpretFalse(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String,List<Object>>()).put("key0", Arrays.asList((Object)"a", "b", "c")).get());
		
		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getIntepretCount());
		assertEquals("key0", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
	}

	public void testInterpretTrue(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new ValueLoopValidator(ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String,List<Object>>()).put("key0", Arrays.asList((Object)"a", "b", "c")).get());
		
		assertTrue(expression.interpret(context));
		assertEquals(3, counter0.getIntepretCount());
		{
			final ValidatorContext tmpContext = counter0.getContextList().removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(2, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList().removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(1, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList().removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(0, tmpContext.getTmpIndex());
		}
	}

	public void testInterpretParse(){
		final ValidatorExpression expression = new ValueLoopValidator(ValidatorConstants.ANDAND, 
				new ParseValidator(new RegExpFormat("[ab]", "\\*")));
		
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String,List<Object>>()).put("key0", Arrays.asList((Object)"a", "b", "c")).get());
		
		assertTrue(expression.interpret(context));
		assertTrue(TestUtils.equals(MapBuilder.get(new HashMap<String,Object>()).put("key0", Arrays.asList("*","*","c")).get(), context.getInputMap()));
	}

	public void testSimpleConstractor(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String,List<Object>>()).put("key0", Arrays.asList((Object)"a", "b", "c")).get());
		
		assertFalse(expression.interpret(context));
		assertEquals(3, counter0.getIntepretCount());
		{
			final ValidatorContext tmpContext = counter0.getContextList().removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(2, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList().removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(1, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList().removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(0, tmpContext.getTmpIndex());
		}
	}
	
	public void testAppendTo() throws IOException{
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(ValidatorConstants.ANDAND, counter0);
		final StringBuilder sb = new StringBuilder();
		expression.appendTo(sb);
		assertEquals("new ValueLoopValidator(new AndAndValidatorIterable(),new FalseValidator())", sb.toString());
		assertEquals(1, counter0.getAppendToCount());
		assertTrue(sb == counter0.getSbList().getLast());
	}
	
	public void testAddMessageTo(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(ValidatorConstants.ANDAND, counter0);
		final Collection<String> expectedCollection = new ArrayList<String>();
		expression.addMessageTo(expectedCollection);
		assertEquals(1, counter0.getAddMessageToCount());
		assertTrue(expectedCollection == counter0.getCollectionList().getLast());
	}
}
