/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;

import junit.framework.TestCase;
import woolpack.utils.SwitchBuilder;

public class BranchByIdValidatorTest extends TestCase {

	public void testConstructor(){
		try{
			new BranchByIdValidator(null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testNotMatch(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByIdValidator(
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setId("z");
		assertTrue(expression.interpret(context));
		assertEquals(0, counter0.getIntepretCount());
		assertEquals(0, counter1.getIntepretCount());
	}

	public void testMatchFalse(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByIdValidator(
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setId("a");
		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getIntepretCount());
		assertEquals(0, counter1.getIntepretCount());
		assertTrue(context == counter0.getContextList().getLast());
	}

	public void testMatchTrue(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorExpression expression = new BranchByIdValidator(
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final ValidatorContext context = new ValidatorContext();
		context.setId("b");
		assertTrue(expression.interpret(context));
		assertEquals(0, counter0.getIntepretCount());
		assertEquals(1, counter1.getIntepretCount());
		assertTrue(context == counter1.getContextList().getLast());
	}
	
	public void testAppendTo() throws IOException{
		final CountValidator counter0 = new CountValidator(ValidatorConstants.TRUE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new BranchByIdValidator(
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final StringBuilder sb = new StringBuilder();
		expression.appendTo(sb);
		assertEquals("new BranchByIdValidator({\"a\":new TrueValidator(),\"b\":new FalseValidator()})", sb.toString());
		assertEquals(1, counter0.getAppendToCount());
		assertEquals(1, counter1.getAppendToCount());
		assertTrue(sb == counter0.getSbList().getLast());
		assertTrue(sb == counter1.getSbList().getLast());
	}
	
	public void testAddMessageTo(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.FALSE);
		final ValidatorExpression expression = new BranchByIdValidator(
				new SwitchBuilder<String,ValidatorExpression>()
				.put("a", counter0)
				.put("b", counter1)
				.get());
		final Collection<String> expectedCollection = new ArrayList<String>();
		expression.addMessageTo(expectedCollection);
		assertEquals(1, counter0.getAddMessageToCount());
		assertEquals(1, counter1.getAddMessageToCount());
		assertTrue(expectedCollection == counter0.getCollectionList().getLast());
		assertTrue(expectedCollection == counter1.getCollectionList().getLast());
	}
}
