/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.LinkedHashMap;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Serial;
import woolpack.dom.XPathAttrValue;
import woolpack.el.ELConstants;
import woolpack.el.FixEL;
import woolpack.el.PathEL;
import woolpack.utils.MapBuilder;

public class UpdateValueTest extends TestCase {

	public void testConstructor(){
		try{
			new UpdateValue(null, ELConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new UpdateValue(ELConstants.NULL, null);
		}catch(final NullPointerException expected){
			fail();
		}
	}
	
	public void testOptionSelect() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm(
						"<OPTION id=\"name0\" selected=\"true\" value=\"value0\" >label0</OPTION>"+
						"<OPTION id=\"name0\" value=\"value1\" >label1</OPTION>"),
				new XPathAttrValue("id", "name0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, 
				"<OPTION id=\"name0\" value=\"value0\" >label0</OPTION>"+
				"<OPTION id=\"name0\" value=\"value1\" selected=\"true\" >label1</OPTION>"));
	}
	
	public void testOptionUpdate() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm(
						"<OPTION id=\"name0\" value=\"value0\" >label0</OPTION>"),
				new XPathAttrValue("id", "name0", new UpdateValue(new FixEL("value1"), null, false)));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<OPTION id=\"name0\" value=\"value1\" >label0</OPTION>"));
	}
	
	public void testRadioSelect() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm(
						"<INPUT name=\"name0\" type=\"radio\" value=\"value0\" checked=\"true\" />"+
						"<INPUT name=\"name0\" type=\"radio\" value=\"value1\" />"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, 
				"<INPUT name=\"name0\" type=\"radio\" value=\"value0\" />"+
				"<INPUT name=\"name0\" type=\"radio\" value=\"value1\" checked=\"true\" />"));
	}
	
	public void testRadioUpdate() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT name=\"name0\" type=\"radio\" value=\"value0\" />"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"), null, false)));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT name=\"name0\" type=\"radio\" value=\"value1\" />"));
	}
	
	public void testInput() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT name=\"name0\" value=\"value0\" />"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT name=\"name0\" value=\"value1\" />"));
	}

	public void testTextArea() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<TEXTAREA name=\"name0\" >value0</TEXTAREA>"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<TEXTAREA name=\"name0\" >value1</TEXTAREA>"));
	}
	
	public void testDiv() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody("<DIV id=\"id0\" >value0</DIV>"),
				new XPathAttrValue("id", "id0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<DIV id=\"id0\" >value1</DIV>"));
	}
	
	public void testSpan() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody("<SPAN id=\"id0\" >value0</SPAN>"),
				new XPathAttrValue("id", "id0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<SPAN id=\"id0\" >value1</SPAN>"));
	}
	
	public void testSpanSelect() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody("<SPAN id=\"id0\" >value0</SPAN>"),
				new XPathAttrValue("id", "id0", new UpdateValue(new FixEL("1"), new PathEL("local.list0"))));
		final DomContext context = new DomContext();
		context.getLocal().put("list0", 
				MapBuilder.get(new LinkedHashMap<String,Object>())
				.put("0", "a")
				.put("1", "b")
				.put("2", "c")
				.get());
		
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<SPAN id=\"id0\" >b</SPAN>"));
	}
	
	public void testFileSelect() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT type=\"file\" name=\"name0\" />"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT type=\"file\" name=\"name0\" />"));
	}
	
	public void testFileUpdate() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT type=\"file\" name=\"name0\" />"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"), null, false)));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT type=\"file\" name=\"name0\" />"));
	}
	
	public void testImageSelect() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT type=\"image\" name=\"name0\" />"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"))));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT type=\"image\" name=\"name0\" />"));
	}
	
	public void testImageUpdate() {
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm("<INPUT type=\"image\" name=\"name0\" />"),
				new XPathAttrValue("name", "name0", new UpdateValue(new FixEL("value1"), null, false)));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, "<INPUT type=\"image\" name=\"name0\" />"));
	}
}
