/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.dom.Count;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.FormatId;
import woolpack.el.EL;
import woolpack.el.PathEL;
import woolpack.text.FixFormat;
import woolpack.utils.EmbeddingResolver;
import woolpack.utils.MapBuilder;
import woolpack.validator.CountValidator;
import woolpack.validator.MessageValidator;
import woolpack.validator.MinLengthValidator;
import woolpack.validator.TmpIndexValidator;
import woolpack.validator.TmpKeyValidator;
import woolpack.validator.ValidatorConstants;
import woolpack.validator.ValidatorExpression;

public class ValidatorBuilderTest extends TestCase {

	public void testConstructor(){
		try{
			new ValidatorBuilder(null, new PathEL("local.validatorError"), true);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new ValidatorBuilder(ValidatorConstants.TRUE, null, true);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testGet(){
		final EL el = new PathEL("local.validatorError");
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(ValidatorConstants.TRUE, el, true);
		assertEquals(el, validatorBuilder.getMessageEL());
	}
	
	public void testException(){
		final ValidatorExpression validatorExpression = new TmpKeyValidator("key0", new TmpIndexValidator(2, new MinLengthValidator(1)));
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(validatorExpression);
		
		final DomExpression expression = validatorBuilder.getCheckExpression(DomConstants.NULL, DomConstants.NULL);
		final DomContext context = new DomContext();
		context.setId("id0");
		context.setInput(MapBuilder.get(new HashMap<String,Object>()).put("key0", new String[]{"value0", "value1"}).get());
		try{
			expression.interpret(context);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testGetCheckExpressionTrue(){
		final CountValidator counter = new CountValidator(ValidatorConstants.TRUE);
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(counter);
		
		final Count counter0 = new Count(new FormatId(new FixFormat("newId")));
		final Count counter1 = new Count(DomConstants.NULL);
		final DomExpression expression = validatorBuilder.getCheckExpression(counter0, counter1);
		final DomContext context = new DomContext();
		context.setId("id0");
		context.setInput(MapBuilder.get(new HashMap<String,Object>()).put("key0", Arrays.asList("value0", "value1")).get());
		expression.interpret(context);
		assertEquals("newId", context.getId());
		assertTrue(context.getLocal().isEmpty());
		assertEquals(1, counter0.getCount());
		assertTrue(context == counter0.getContextList().getLast());
		assertTrue(TestUtils.equals(
				MapBuilder.get(new HashMap<String,Object>())
				.put("key0", Arrays.asList("value0", "value1"))
				.get()
				, counter0.getContextList().getLast().getInput()));
		assertEquals(0, counter1.getCount());
	}
	
	public void testGetCheckExpressionFalse(){
		final CountValidator counter = new CountValidator(new MessageValidator("message0"));
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(counter);
		
		final Count counter0 = new Count(DomConstants.NULL);
		final Count counter1 = new Count(new FormatId(new FixFormat("newId")));
		final DomExpression expression = validatorBuilder.getCheckExpression(counter0, counter1);
		final DomContext context = new DomContext();
		context.setId("id0");
		context.setInput(new HashMap());
		expression.interpret(context);
		assertEquals("newId", context.getId());
		final Object list = ValidatorBuilder.DEFALT_MESSAGES_EL.getValue(context);
		assertTrue(TestUtils.equals(Arrays.asList("message0"), list));
		assertEquals(0, counter0.getCount());
		assertEquals(1, counter1.getCount());
		assertTrue(context == counter1.getContextList().getLast());
	}
	
	
	public void testGetReplaceExpressionTrue(){
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(ValidatorConstants.TRUE, ValidatorBuilder.DEFALT_MESSAGES_EL, true);
		final DomExpression expression = validatorBuilder.getReplaceExpression();
		final DomContext context = new DomContext();
		context.setInput(new HashMap());
		context.setConfig(MapBuilder.get(new HashMap<String,Object>()).put("messageKey0", "messageValue0").get());
		{
			final List<String> list = new ArrayList<String>();
			list.add("messageKey0");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("messageValue0"), list));
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("messageKey1");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("messageKey1"), list));
		}
	}
	
	public void testGetMessageExpressionFalse(){
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(ValidatorConstants.TRUE);
		final DomExpression expression = validatorBuilder.getReplaceExpression();
		final DomContext context = new DomContext();
		context.setConfig(MapBuilder.get(new HashMap<String,Object>()).put("messageKey0", "messageValue0").get());
		{
			final List<String> list = new ArrayList<String>();
			list.add("messageKey0");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("messageValue0"), list));
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("messageKey0");
			list.add("messageKey1");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			try{
				expression.interpret(context);
				fail();
			}catch(final NullPointerException e){
				
			}
			assertEquals(2, list.size());
			assertTrue(TestUtils.equals(Arrays.asList("messageValue0", "messageKey1"), list));
		}
	}

	
	public void testResolveEmbeddingExpressionTrue(){
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(ValidatorConstants.TRUE, ValidatorBuilder.DEFALT_MESSAGES_EL, true);
		final DomExpression expression = validatorBuilder.getResolveEmbeddingExpression(new EmbeddingResolver());
		final DomContext context = new DomContext();
		{
			final Map<String,Object> map = new HashMap<String,Object>();
			map.put("baseKey0", "baseValue0");
			map.put("baseKey1", "baseValue1");
			map.put("abstractKey0", "a{0}b{1}c");
			context.setConfig(map);
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("baseKey0");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("baseValue0"), list));
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("#{abstractKey0}{baseKey0}{baseKey1}");
			list.add("x{baseKey0}y{baseKey1}z");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("abaseValue0bbaseValue1c", "xbaseValue0ybaseValue1z"), list));
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("baseKey2");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("baseKey2"), list));
		}
	}
	
	public void testResolveEmbeddingExpressionFalse(){
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(ValidatorConstants.TRUE);
		final DomExpression expression = validatorBuilder.getResolveEmbeddingExpression(new EmbeddingResolver());
		final DomContext context = new DomContext();
		{
			final Map<String,Object> map = new HashMap<String,Object>();
			map.put("baseKey0", "baseValue0");
			map.put("baseKey1", "baseValue1");
			map.put("abstractKey0", "a{0}b{1}c");
			context.setConfig(map);
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("baseKey0");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("baseValue0"), list));
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("#{abstractKey0}{baseKey0}{baseKey1}");
			list.add("x{baseKey0}y{baseKey1}z");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			expression.interpret(context);
			assertTrue(TestUtils.equals(Arrays.asList("abaseValue0bbaseValue1c", "xbaseValue0ybaseValue1z"), list));
		}
		{
			final List<String> list = new ArrayList<String>();
			list.add("baseKey0");
			list.add("baseKey2");
			ValidatorBuilder.DEFALT_MESSAGES_EL.setValue(context, list);
			try{
				expression.interpret(context);
				fail();
			}catch(final NullPointerException e){
				
			}
			assertTrue(TestUtils.equals(Arrays.asList("baseValue0", "baseKey2"), list));
		}
	}
}
