/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * リフレクションを使用した{@link PropertyELFactory}。
 * getter/setterのオーバライドには対応していない。
 * @author nakamura
 *
 */
public class ReflectionPropertyELFactory implements PropertyELFactory {
	private final GettingEL convertEL;
	
	/**
	 * コンストラクタ。
	 * @param convertEL 型変換器。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public ReflectionPropertyELFactory(final GettingEL convertEL){
		convertEL.getClass();
		this.convertEL = convertEL;
	}
	
	private static boolean equalsStoic(final String s0, final String s1){
		return (s0 == s1) || s0.equals(s1);
	}
	
	private static boolean equals(final String s0, final String s1){
		return equalsStoic(s0, s1) || (
				(Character.toUpperCase(s0.charAt(0)) == Character.toUpperCase(s1.charAt(0))) && 
				equalsStoic(s0.substring(1), s1.substring(1))
		);
	}

	public GettingEL newGetter(final Class clazz, final String propertyName) {
		final Method[] methList = clazz.getMethods();
		for (int i=0; i<methList.length; i++) {
			final Method method = methList[i];
			{
				final Class[] paramTypes = method.getParameterTypes();
				if(paramTypes != null && paramTypes.length > 0){
					continue;
				}
			}
			if(Void.TYPE.equals(method.getReturnType())){
				continue;
			}
			final String key = method.getName();
			if(
					(key.startsWith("get") && equals(key.substring(3), propertyName)) || 
					(key.startsWith("is") && equals(key.substring(2), propertyName))
					){
				return new GettingELAdapter(){
					@Override public Object getValue(final Object root, final Class toType)
							throws ELTargetRuntimeException {
						try {
							final Object result = method.invoke(root, new Object[0]);
							return convertEL.getValue(result, toType);
						} catch (final IllegalArgumentException e) {
							throw new RuntimeException(e);
						} catch (final IllegalAccessException e) {
							throw new RuntimeException(e);
						} catch (final InvocationTargetException e) {
							throw new ELTargetRuntimeException(e);
						}
					}
				};
			}
		}
		throw new RuntimeException("property not found: " + clazz + ": " + propertyName);
	}

	public EL newSetter(final Class clazz, final String propertyName) {
		final Method[] methList = clazz.getMethods();
		for (int i=0; i<methList.length; i++) {
			final Method method = methList[i];
			{
				final Class[] paramTypes = method.getParameterTypes();
				if(paramTypes != null && paramTypes.length != 1){
					continue;
				}
			}
			final String key = method.getName();
			if(key.startsWith("set") && equals(key.substring(3), propertyName)){
				return new ELAdapter(){	
					@Override public void setValue(final Object root, final Object value) throws ELTargetRuntimeException {
						try {
							method.invoke(root, new Object[]{convertEL.getValue(value, method.getParameterTypes()[0])});
						} catch (final IllegalArgumentException e) {
							throw new RuntimeException(e);
						} catch (final IllegalAccessException e) {
							throw new RuntimeException(e);
						} catch (final InvocationTargetException e) {
							throw new ELTargetRuntimeException(e);
						}
					}
					@Override
					public Object getValue(Object root, Class clazz) throws ELTargetRuntimeException {
						return null;// カバレージがここを通過してはいけない
					}
				};
			}
		}
		throw new RuntimeException("property not found: " + clazz + ": " + propertyName);
	}
	
	@Override public int hashCode(){
		return new HashCodeBuilder().append(convertEL).toHashCode();
	}
	
	@Override public boolean equals(final Object obj){
		if(!(obj instanceof ReflectionPropertyELFactory)){
			return false;
		}
		if(this == obj){
			return true;
		}
		final ReflectionPropertyELFactory o = (ReflectionPropertyELFactory)obj;
		return new EqualsBuilder()
		.append(this.convertEL, o.convertEL)
		.isEquals();
	}
	
	@Override public String toString(){
		return new ToStringBuilder(this).append("convertEL", convertEL).toString();
	}
}
